#include "metabase_shard_manager.h"

#include <library/cpp/json/json_reader.h>
#include <util/string/cast.h>
#include <util/string/join.h>

using namespace NHistDb::NStockpile;

namespace NHistDb::NStockpile {
    TString DetectBaseUrl(EStockpileClusterType clusterType) {
        switch (clusterType) {
            case EStockpileClusterType::Production: {
                return "http://solomon.yandex.net/api/v2";
            }
            case EStockpileClusterType::Testing: {
                return "http://solomon-test.yandex.net/api/v2";
            }
            case EStockpileClusterType::Prestable: {
                return "http://solomon-prestable.yandex.net/api/v2";
            }
        }
    }
}

TMetabaseShardManager::TMetabaseShardManager(const TSettings& settings)
    : Settings(settings)
    , BaseUrl(DetectBaseUrl(Settings.ClusterInfo.ClusterType))
{
}

NJson::TJsonValue TMetabaseShardManager::CheckForAlreadyExistError(
    NHttpFetcher::TResultRef response
) const {
    if (response->Code == 409) {
        NJson::TJsonValue responseJson;
        NJson::ReadJsonTree(response->Data, &responseJson, true);
        return responseJson;
    } else {
        ythrow TShardCreationError() << "[" << response->Code << "]: " << response->Data;
    }
}

NJson::TJsonValue TMetabaseShardManager::MakeRequest(
    const TString& projectPath,
    const TMaybe<TString>& requestJson,
    const TDuration& timeout,
    bool existIsOk
) const {
    if (Settings.AuthToken.empty()) {
        ythrow TShardCreationError() << "no OAuth token provided, can't perform stockpile API request";
    }

    TString url = Join("/", BaseUrl, "projects", projectPath);

    NHttp::TFetchOptions options{};
    options.SetOAuthToken(Settings.AuthToken);
    options.SetContentType({"application/json"});
    options.SetUserAgent(YASM_USER_AGENT);
    options.SetPostData(requestJson);
    options.SetTimeout(timeout);
    NHttp::TFetchQuery query{url, options};

    auto response = NHttp::Fetch(query);
    if (response->Success()) {
        NJson::TJsonValue responseJson;
        NJson::ReadJsonTree(response->Data, &responseJson, true);
        return responseJson;
    } else {
        if (existIsOk) {
            return CheckForAlreadyExistError(response);
        } else {
            ythrow TShardCreationError() << "[" << response->Code << "]: " << response->Data;
        }
    }
}

void TMetabaseShardManager::CreateProject(const TMetabaseShardKey& shardKey) const {
    NJsonWriter::TBuf requestJson{};
    NJsonWriter::TPairContext pairContext {requestJson.BeginObject()};
    pairContext
        .WriteKey("id")
        .WriteString(shardKey.GetProjectId())
        .WriteKey("name")
        .WriteString(shardKey.GetProject())
        .WriteKey("abcService")
        .WriteString(DEFAULT_ABC_SERVICE);
    pairContext.EndObject();
    MakeRequest("", {requestJson.Str()});
}

void TMetabaseShardManager::CreateCluster(const TMetabaseShardKey& shardKey) const {
    NJsonWriter::TBuf requestJson{};
    requestJson.BeginObject()
        .WriteKey("id")
        .WriteString(shardKey.GetClusterId())
        .WriteKey("name")
        .WriteString(shardKey.GetCluster());
    if (shardKey.IsGroupShard()) {
        requestJson.WriteKey("sensorsTtlDays").WriteInt(0);
    } else {
        requestJson.WriteKey("sensorsTtlDays").WriteInt(30);
    }
    requestJson.EndObject();
    MakeRequest(Join("/", shardKey.GetProjectId(), "clusters"), {requestJson.Str()});
}

void TMetabaseShardManager::CreateService(const TMetabaseShardKey& shardKey) const {
    NJsonWriter::TBuf requestJson{};
    requestJson.BeginObject()
        .WriteKey("id")
        .WriteString(shardKey.GetServiceId())
        .WriteKey("name")
        .WriteString(shardKey.GetService())
        .WriteKey("gridSec")
        .WriteInt(5)
        .EndObject();

    MakeRequest(Join("/", shardKey.GetProjectId(), "services"), {requestJson.Str()});
}

void TMetabaseShardManager::CreateShard(const TMetabaseShardKey& shardKey) const {
    NJsonWriter::TBuf requestJson{};
    requestJson.BeginObject()
        .WriteKey("clusterId")
        .WriteString(shardKey.GetClusterId())
        .WriteKey("serviceId")
        .WriteString(shardKey.GetServiceId())
        .WriteKey("id")
        .WriteString(shardKey.GetShardId())
        .WriteKey("sensorNameLabel")
        .WriteString(SIGNAL_LABEL_KEY)
        .WriteKey("decimPolicy")
        .WriteString(POLICY_5_MIN_AFTER_8_DAYS_NAME)
        .EndObject();
    MakeRequest(Join("/", shardKey.GetProjectId(), "shards"), {requestJson.Str()});
}
