#include "metabase_shard_provider.h"
#include "metabase_client.h"
#include "metrics.h"

#include <infra/monitoring/common/perf.h>

#include <util/generic/list.h>
#include <util/random/random.h>

using namespace NHistDb::NStockpile;

namespace {
    TDuration GetDurationJitter(TDuration jitterInterval) {
        return TDuration::FromValue(RandomNumber(jitterInterval.GetValue()));
    }
}

TStockpileMetabaseAsyncShardProvider::TStockpileMetabaseAsyncShardProvider(
        TClusterProvider& clusterProvider,
        ITopologyListener& listener,
        const TSettings& settings,
        TLog& log)
    : ClusterProvider(clusterProvider)
    , Listener(listener)
    , Settings(settings)
    , MetabaseShardManager(Settings)
    , Log(log)
    , LastUpdate(0)
    , LastStatsUpdate(0)
    , LastApiBackoff()
    , DontRequestApiBefore(TInstant::Zero())
{
}

void TStockpileMetabaseAsyncShardProvider::TryToUpdateShards() {
    auto now = TInstant::Now();
    if (now - TInstant::FromValue(static_cast<TInstant::TValue>(AtomicGet(LastUpdate))) >= METABASE_UPDATE_INTERVAL) {
        UpdateShardsOnHosts();
        UpdateShardStats();
    } else {
        if (now - TInstant::FromValue(static_cast<TInstant::TValue>(AtomicGet(LastStatsUpdate))) >= STATS_UPDATE_INTERVAL) {
            UpdateShardStats();
        }
    }
}

void TStockpileMetabaseAsyncShardProvider::UpdateShardStats() {
    TLightReadGuard guard(EntriesLock);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_METABASE_CACHE_SIZE, ShardCache.size());
    size_t sensorCacheSize = 0;
    size_t missingShards = 0;
    size_t sensorsInMissingShards = 0;
    size_t shardsInCreationQueue = 0;
    for (const auto& pair : ShardCache) {
        auto& entry = pair.second;
        if (!entry.CreationQueueEntry.Empty()) {
            ++shardsInCreationQueue;
        }
        if (!entry.State.GrpcRemoteHost) {
            ++missingShards;
            sensorsInMissingShards += entry.State.Shard->SensorCacheSize();
        }

        sensorCacheSize += entry.State.Shard->SensorCacheSize();
    }
    TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_MISSING_METABASE_SHARDS, missingShards);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_METABASE_SHARDS_IN_CREATION_QUEUE, shardsInCreationQueue);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_SENSOR_CACHE_SIZE, sensorCacheSize);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_SENSORS_IN_MISSING_SHARDS, sensorsInMissingShards);

    AtomicSet(LastStatsUpdate, static_cast<TAtomicBase>(TInstant::Now().GetValue()));
}

TMaybe<TMetabaseShardState> TStockpileMetabaseAsyncShardProvider::FindOneShard(const TMetabaseShardKey& shardKey) const {
    TMaybe<TMetabaseShardState> result;
    TLightReadGuard guard(EntriesLock);
    auto it = ShardCache.find(shardKey);
    if (it != ShardCache.end()) {
        result.ConstructInPlace(it->second.State);
    }
    return result;
}

TMetabaseShardState TStockpileMetabaseAsyncShardProvider::ResolveOneShard(const TMetabaseShardKey& shardKey) {
    auto state = FindOneShard(shardKey);
    if (state.Defined()) {
        return state.GetRef();
    } else {
        TLightWriteGuard guard(EntriesLock);
        TMetabaseShards::insert_ctx ctx;
        auto it = ShardCache.find(shardKey, ctx);
        if (it == ShardCache.end()) {
            if (!Settings.CreateNewShards) {
                throw TShardNotFoundError() << "Will not create shard " << shardKey << ". Shard creation is disabled in settings.";
            }
            it = ShardCache.emplace_direct(ctx, shardKey, MakeAtomicShared<TMetabaseShard>(shardKey));
            ShardCreationQueue.PushBack(&it->second.CreationQueueEntry);
            it->second.State.NotReadySince = TInstant::Now();
        }
        return it->second.State;
    }
}

TVector<TAtomicSharedPtr<TMetabaseShard>> TStockpileMetabaseAsyncShardProvider::GetShards() const {
    TLightReadGuard guard(EntriesLock);
    TVector<TAtomicSharedPtr<TMetabaseShard>> shards(Reserve(ShardCache.size()));
    for (const auto& [key, state] : ShardCache) {
        shards.emplace_back(state.State.Shard);
    }
    return shards;
}

TVector<TMetabaseShardKey> TStockpileMetabaseAsyncShardProvider::GetShardKeys(bool skipUnavailable) const {
    TLightReadGuard guard(EntriesLock);
    TVector<TMetabaseShardKey> shardKeys(Reserve(ShardCache.size()));
    for (const auto& [key, state] : ShardCache) {
        if (!skipUnavailable || state.State.GrpcRemoteHost) {
            shardKeys.emplace_back(key);
        }
    }
    return shardKeys;
}

THashSet<TString> TStockpileMetabaseAsyncShardProvider::GetItypes() const {
    THashSet<TString> itypes;
    {
        TLightReadGuard guard(EntriesLock);
        for (const auto& [key, state] : ShardCache) {
            TStringBuf projectId = key.GetProjectId();
            if (projectId.StartsWith(STOCKPILE_YASM_PROJECTS_PREFIX)) {
                TStringBuf itype = projectId.SubStr(STOCKPILE_YASM_PROJECTS_PREFIX.size());
                itypes.insert(TString{itype});
            }
        }
    }
    return itypes;
}

void TStockpileMetabaseAsyncShardProvider::UpdateProjectsCache() {
    auto shardKeys = GetShardKeys(true);

    TGuard guard(ProjectsLock);
    for (const auto& shardKey : shardKeys) {
        auto& cacheItem = ExistingProjectsCache[shardKey.GetProjectId()];
        cacheItem.ClusterCache.insert(shardKey.GetCluster());
        cacheItem.ServiceCache.insert(shardKey.GetService());
    }
}

bool TStockpileMetabaseAsyncShardProvider::CreateOneShard(TMetabaseShardKey shardKey) {
    bool result = false;
    size_t operationsPerformed = 0;
    try {
        TGuard guard(ProjectsLock);

        THashMap<TString, TProjectCacheItem>::insert_ctx ctx;

        const auto& projectId = shardKey.GetProjectId();
        auto it = ExistingProjectsCache.find(projectId, ctx);

        if (it.IsEnd()) {
            MetabaseShardManager.CreateProject(shardKey);
            ++operationsPerformed;
            it = ExistingProjectsCache.emplace_direct(ctx, projectId, TProjectCacheItem{});
        }
        auto& projectCacheItem = it->second;
        if (!projectCacheItem.ClusterCache.contains(shardKey.GetCluster())) {
            MetabaseShardManager.CreateCluster(shardKey);
            ++operationsPerformed;
            projectCacheItem.ClusterCache.emplace(shardKey.GetCluster());
        }
        if (!projectCacheItem.ServiceCache.contains(shardKey.GetService())) {
            MetabaseShardManager.CreateService(shardKey);
            ++operationsPerformed;
            projectCacheItem.ServiceCache.emplace(shardKey.GetService());
        }
        MetabaseShardManager.CreateShard(shardKey);
        ++operationsPerformed;
        result = true;
    } catch (const TShardCreationError& exc) {
        Log << TLOG_WARNING << "Failed to create shard " << shardKey << ": " << exc.what();
        TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_METABASE_SHARD_CREATION_ERROR, 1);
    }
    TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_METABASE_SHARD_CREATION_OPERATIONS, operationsPerformed);
    return result;
}

void TStockpileMetabaseAsyncShardProvider::UpdateShardsOnHosts() {
    TGrpcStateHandler handler(Log);
    TList<TMetabaseStatusState> states;
    for (const auto& host : ClusterProvider.GetHosts()) {
        states.emplace_back(handler.GetQueue(), host, Log);
    }
    handler.Wait();

    auto changedShards = ProcessShardsStates(states);
    for (const auto& shardKey : changedShards) {
        Listener.OnChangedMetabaseShard(shardKey);
    }

    UpdateProjectsCache();

    AtomicSet(LastUpdate, static_cast<TAtomicBase>((TInstant::Now() - GetDurationJitter(METABASE_UPDATE_JITTER)).GetValue()));
}

TVector<TMetabaseShardKey> TStockpileMetabaseAsyncShardProvider::ProcessShardsStates(TList<TMetabaseStatusState>& states) {
    TVector<TMetabaseShardKey> changedShards;
    THashSet<TMetabaseShardKey> availableShards;
    auto now = TInstant::Now();

    for (auto& state : states) {
        try {
            const auto foundShards = state.GetResult();
            auto metabaseHost = state.GetHost();

            Log << TLOG_DEBUG << foundShards.size() << " metabase shards on " << *state.GetHost() << " found";
            for (auto& shardStatus: foundShards) {
                auto shardKey = shardStatus.Key;
                if (availableShards.insert(shardKey).second) {
                    TLightWriteGuard guard(EntriesLock);

                    TMetabaseShards::insert_ctx ctx;
                    auto it = ShardCache.find(shardKey, ctx);
                    bool isNewShard = it == ShardCache.end();
                    if (isNewShard) {
                        it = ShardCache.emplace_direct(ctx, shardKey, MakeAtomicShared<TMetabaseShard>(shardKey, shardStatus.NumId));
                    }
                    it->second.CreationQueueEntry.Unlink(); // remove from creation queue
                    auto& curShardState = it->second.State;

                    bool shardRecreated = curShardState.Shard->GetShardNumId() != shardStatus.NumId;
                    if (shardRecreated) {
                        curShardState.Shard = MakeAtomicShared<TMetabaseShard>(shardKey, shardStatus.NumId);
                    }

                    bool shardChanged = curShardState.GrpcRemoteHost != metabaseHost ||
                                        curShardState.IsReady() != shardStatus.Ready ||
                                        curShardState.ReadOnly != !shardStatus.ReadyWrite;
                    curShardState.GrpcRemoteHost = metabaseHost;
                    curShardState.ReadOnly = !shardStatus.ReadyWrite;

                    if (shardStatus.Ready && curShardState.NotReadySince.Defined()) {
                        curShardState.NotReadySince.Clear();
                    } else if (!shardStatus.Ready && !curShardState.NotReadySince.Defined()) {
                        curShardState.NotReadySince = now;
                    }

                    if (isNewShard) {
                        changedShards.push_back(shardKey);
                        Log << TLOG_INFO << "New metabase shard " << curShardState;
                    } else if (shardRecreated) {
                        changedShards.push_back(shardKey);
                        Log << TLOG_WARNING << "Recreated metabase shard " << curShardState << ". Sensor cache cleared";
                    } else if (shardChanged) {
                        changedShards.push_back(shardKey);
                        Log << TLOG_INFO << "Changed metabase shard " << curShardState;
                    }

                    if (curShardState.IsReady()) {
                        if (!curShardState.ReadOnly) {
                            curShardState.Shard->ClearRejectedSensors();
                        }
                        curShardState.Shard->ClearSensorCache(SHARD_SENSOR_FULL_CACHE_REFRESH_INTERVAL,
                            SHARD_SENSOR_CACHE_CLEANUP_INTERVAL + GetDurationJitter(SHARD_SENSOR_CACHE_CLEANUP_JITTER));
                    }
                }
            }

        } catch (...) {
            Log << TLOG_ERR << "Metabase host " << *state.GetHost() << " not working: " << CurrentExceptionMessage();
        }
    }
    {
        TLightWriteGuard guard(EntriesLock);
        for (auto& [key, entry]: ShardCache) {
            if (!availableShards.contains(key)) {
                entry.State.GrpcRemoteHost.Reset();
                if (!entry.State.NotReadySince.Defined()) {
                    entry.State.NotReadySince = now;
                }
                Log << TLOG_WARNING << "Metabase shard " << entry.State << " is unavailable since " << entry.State.NotReadySince;
            }
        }
        Log << TLOG_INFO << "There are " << ShardCache.size() << " metabase shards, " << changedShards.size()
            << " changed, " << availableShards.size() << " received.";
    }

    return changedShards;
}

void TStockpileMetabaseAsyncShardProvider::ProcessShardCreationQueue(TInstant deadline) {
    try {
        while (DontRequestApiBefore < deadline && TInstant::Now() < deadline) {
            SleepUntil(DontRequestApiBefore);

            TMetabaseShardKey shardKey;
            {
                TLightReadGuard guard(EntriesLock);
                if (ShardCreationQueue.Empty()) {
                    return;
                }
                shardKey = ShardCreationQueue.Front()->ParentEntry.State.Shard->GetShardKey();
            }
            auto created = CreateOneShard(shardKey);
            {
                TLightWriteGuard guard(EntriesLock);
                auto it = ShardCache.find(shardKey);
                if (it != ShardCache.end()) {
                    it->second.CreationQueueEntry.Unlink();
                    if (!created) {
                        // move to the end of the queue
                        ShardCreationQueue.PushBack(&it->second.CreationQueueEntry);
                    }
                }
            }
            if (created) {
                LastApiBackoff.Clear();
            } else {
                LastApiBackoff = (LastApiBackoff.Defined()) ?
                    Min(LastApiBackoff.GetRef() * 2, SOLOMON_API_V2_REQUEST_MAX_BACKOFF):
                    SOLOMON_API_V2_REQUEST_INITIAL_BACKOFF;
                DontRequestApiBefore = TInstant::Now() + LastApiBackoff.GetRef();
            }
        }
    } catch (...) {
        TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_METABASE_SHARD_CREATION_ERROR, 1);
        Log << TLOG_ERR << "Unexpected error when trying to process shard creation queue :" << CurrentExceptionMessage();
    }
}

template <>
void Out<TMetabaseShardState>(IOutputStream& stream, TTypeTraits<TMetabaseShardState>::TFuncParam shardState) {
    stream << "TMetabaseShardState(Key=" << shardState.Shard->GetShardKey() << ", Host=";
    if (shardState.GrpcRemoteHost) {
        stream << *shardState.GrpcRemoteHost;
    } else {
        stream << "UNKNOWN";
    }
    stream << ", Ready=" << shardState.IsReady() << ", ReadOnly=" << shardState.ReadOnly << ")";
}
