#pragma once

#include "metabase_shard.h"
#include "metabase_shard_manager.h"
#include "cluster.h"
#include "listener.h"

#include <infra/yasm/common/labels/tags/instance_key.h>

#include <library/cpp/containers/ring_buffer/ring_buffer.h>
#include <library/cpp/logger/log.h>
#include <library/cpp/threading/light_rw_lock/lightrwlock.h>
#include <util/generic/hash.h>

namespace NHistDb {
    namespace NStockpile {
        static constexpr TDuration METABASE_UPDATE_INTERVAL = TDuration::Seconds(40);
        static constexpr TDuration METABASE_UPDATE_JITTER = TDuration::Seconds(20);
        static constexpr TDuration STATS_UPDATE_INTERVAL = TDuration::Seconds(5);
        static constexpr TDuration SOLOMON_API_V2_REQUEST_INITIAL_BACKOFF = TDuration::Seconds(1);
        static constexpr TDuration SOLOMON_API_V2_REQUEST_MAX_BACKOFF = TDuration::Seconds(5);
        static constexpr TDuration SHARD_SENSOR_FULL_CACHE_REFRESH_INTERVAL = TDuration::Days(10);
        static constexpr TDuration SHARD_SENSOR_CACHE_CLEANUP_INTERVAL = TDuration::Minutes(30);
        static constexpr TDuration SHARD_SENSOR_CACHE_CLEANUP_JITTER = TDuration::Minutes(60);


        class TStockpileReadDisable : public yexception {};
        class TStockpileDumpDisable : public yexception {};

        struct TProjectCacheItem {
            THashSet<TString> ClusterCache;
            THashSet<TString> ServiceCache;
        };

        struct TMetabaseShardState {
            TAtomicSharedPtr<TMetabaseShard> Shard;
            TAtomicSharedPtr<TGrpcRemoteHost> GrpcRemoteHost;
            bool ReadOnly = false;
            TMaybe<TInstant> NotReadySince; // set when shard is not found on any host or is marked not ready by stockpile
                                            // reset when shard is found again on some host and is reported by stockpile as ready
            bool IsReady() const {
                return !NotReadySince.Defined() && GrpcRemoteHost;
            }
        };

        class TStockpileMetabaseAsyncShardProvider {
        public:
            TStockpileMetabaseAsyncShardProvider(
                TClusterProvider& clusterProvider,
                ITopologyListener& listener,
                const TSettings& settings,
                TLog& log
            );

            /**
             * Tries to update shard list by requesting shard list from every metabase host
             * DO NOT call from more than 1 thread concurrently.
             * */
            void TryToUpdateShards();

            /**
             * Process shard creation tasks that were enqueued during ResolveOneShard calls.
             * DO NOT call from more than 1 thread concurrently.
             * @param deadline - the moment when to stop processing next shard creation tasks and return.
             */
            void ProcessShardCreationQueue(TInstant deadline);

            /** Find one shard in caches */
            TMaybe<TMetabaseShardState> FindOneShard(const TMetabaseShardKey& shardKey) const;

            /** Find one shard in caches and if there is none, then start shard creation task in background */
            TMetabaseShardState ResolveOneShard(const TMetabaseShardKey& shardKey);

            THashSet<TString> GetItypes() const;
            TVector<TAtomicSharedPtr<TMetabaseShard>> GetShards() const;
            TVector<TMetabaseShardKey> GetShardKeys(bool skipUnavailable) const;

        private:
            struct TShardEntry;
            struct TCreationQueueEntry: TIntrusiveListItem<TCreationQueueEntry> {
                TShardEntry& ParentEntry;

                explicit TCreationQueueEntry(TShardEntry& parentEntry)
                    : ParentEntry(parentEntry) {
                }
            };
            using TCreationQueue = TIntrusiveList<TCreationQueueEntry>;

            struct TShardEntry {
                TMetabaseShardState State;
                TCreationQueueEntry CreationQueueEntry;

                explicit TShardEntry(TAtomicSharedPtr<TMetabaseShard> shard)
                    : State()
                    , CreationQueueEntry(*this) {
                    State.Shard = std::move(shard);
                }
            };
            using TMetabaseShards = THashMap<TMetabaseShardKey, TShardEntry>;

            void UpdateShardsOnHosts();
            TVector<TMetabaseShardKey> ProcessShardsStates(TList<TMetabaseStatusState>& states);

            bool CreateOneShard(TMetabaseShardKey shardKey);
            void UpdateProjectsCache();

            void UpdateShardStats();

            TClusterProvider& ClusterProvider;
            ITopologyListener& Listener;
            const TSettings& Settings;
            TMetabaseShardManager MetabaseShardManager;
            TLog& Log;

            TAtomic LastUpdate;
            TAtomic LastStatsUpdate;

            TMaybe<TDuration> LastApiBackoff;
            TInstant DontRequestApiBefore;

            TLightRWLock EntriesLock;
            TMetabaseShards ShardCache;
            TCreationQueue ShardCreationQueue;

            TMutex ProjectsLock;
            THashMap<TString, TProjectCacheItem> ExistingProjectsCache;
        };
    }
}
