#include "metabase_shard.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NTags;
using namespace NHistDb::NStockpile;
using namespace NZoom::NSignal;
using namespace NHistDb::NStockpile;
using yandex::solomon::model::MetricType;

Y_UNIT_TEST_SUITE(TMetabaseShardSensorCacheTest) {
    Y_UNIT_TEST(RecordUpdate) {
        TMetabaseShardKey shardKey;
        TMetabaseShard shard(shardKey);
        auto key1 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=prestable"), TSignalName(TString("signal1_summ")));
        auto sensor1 = TSensorId(0, 0, MetricType::DSUMMARY);
        auto key2 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=unstable"), TSignalName(TString("signal2_summ")));
        auto sensor2 = TSensorId(0, 1, MetricType::DSUMMARY);
        shard.SaveSensors({{key1, sensor1}, {key2, sensor2}});

        UNIT_ASSERT_EQUAL(sensor1, shard.FindSensor(key1));
        UNIT_ASSERT_EQUAL(sensor2, shard.FindSensor(key2));

        auto sensor3 = TSensorId(0, 2, MetricType::DSUMMARY);
        shard.SaveSensors({{key2, sensor3}});

        UNIT_ASSERT_EQUAL(sensor1, shard.FindSensor(key1));
        UNIT_ASSERT_EQUAL(sensor3, shard.FindSensor(key2));
    }

    Y_UNIT_TEST(RecordTimestampUpdate) {
        TMetabaseShardKey shardKey;
        TMetabaseShard shard(shardKey, TInstant::Seconds(100));
        auto key1 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=prestable"), TSignalName(TString("signal1_summ")));
        auto sensor1 = TSensorId(0, 0, MetricType::DSUMMARY);
        auto key2 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=unstable"), TSignalName(TString("signal2_summ")));
        auto sensor2 = TSensorId(0, 1, MetricType::DSUMMARY);
        shard.SaveSensors({{key1, sensor1}});
        shard.SaveSensors({{key2, sensor2}});

        UNIT_ASSERT_EQUAL(sensor1, shard.FindSensor(key1));
        UNIT_ASSERT_EQUAL(sensor2, shard.FindSensor(key2));

        auto sensor3 = TSensorId(0, 2, MetricType::DSUMMARY);
        shard.SaveSensors({{key1, sensor3}});

        UNIT_ASSERT_EQUAL(sensor3, shard.FindSensor(key1));
        UNIT_ASSERT_EQUAL(sensor2, shard.FindSensor(key2));

        shard.ClearSensorCache(TDuration::Seconds(200), TDuration::Seconds(100), TInstant::Seconds(200));

        UNIT_ASSERT(!shard.FindSensor(key2));
        UNIT_ASSERT_EQUAL(sensor3, shard.FindSensor(key1));
    }

    Y_UNIT_TEST(PartialCleanup) {
        TMetabaseShardKey shardKey;
        TMetabaseShard shard(shardKey, TInstant::Seconds(100));
        const size_t startSize = 88;
        TVector<TSeriesKey> allKeys;
        for (size_t i = 0; i < startSize; ++i) {
            TStringStream s;
            s << "signal" << i << "_summ";
            auto signalName = TSignalName(s.Str());
            auto key = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=prestable"), signalName);
            allKeys.push_back(key);
            auto sensor = TSensorId(0, i, MetricType::DSUMMARY);
            shard.SaveSensors({{key, sensor}});
        }
        UNIT_ASSERT_VALUES_EQUAL(startSize, shard.SensorCacheSize());

        shard.ClearSensorCache(TDuration::Seconds(400), TDuration::Seconds(100), TInstant::Seconds(200));
        size_t lastSize = startSize * (1 - 0.25);
        UNIT_ASSERT_VALUES_EQUAL(lastSize, shard.SensorCacheSize());

        for (size_t i = 0; i < startSize; ++i) {
            auto key = allKeys[i];
            auto foundSensor = shard.FindSensor(key);
            if (i < startSize - lastSize) {
                UNIT_ASSERT(!foundSensor);
            } else {
                UNIT_ASSERT_EQUAL(TSensorId(0, i, MetricType::DSUMMARY), foundSensor);
            }
        }
    }

    Y_UNIT_TEST(FullRefresh) {
        TMetabaseShardKey shardKey;
        const auto startTime = TInstant::Seconds(100);
        TMetabaseShard shard(shardKey, startTime);
        const size_t size = 80;
        TVector<TSeriesKey> allKeys;
        for (size_t i = 0; i < size; ++i) {
            TStringStream s;
            s << "signal" << i << "_summ";
            auto signalName = TSignalName(s.Str());
            auto key = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=prestable"), signalName);
            allKeys.push_back(key);
        }

        const auto fullRefreshInterval = TDuration::Seconds(400);
        const auto minInterCleanupInterval = TDuration::Seconds(10);

        TVector<TInstant> stepStartTimes({
            startTime,
            startTime + TDuration::Seconds(125),
            startTime + TDuration::Seconds(200),
            startTime + TDuration::Seconds(250),
            startTime + TDuration::Seconds(400)
        });

        size_t nextSensorId = 0;
        for (size_t step = 0; step < 5; ++step) {
            auto stepStartTime = stepStartTimes[step];
            shard.ClearSensorCache(fullRefreshInterval, minInterCleanupInterval, stepStartTime);
            UNIT_ASSERT(size > shard.SensorCacheSize());
            for (const auto& key: allKeys) {
                if (!shard.FindSensor(key)) {
                    auto sensor = TSensorId(step, ++nextSensorId, MetricType::DSUMMARY);
                    shard.SaveSensors({{key, sensor}});
                }
            }
            UNIT_ASSERT_VALUES_EQUAL(size, shard.SensorCacheSize());
            // check no cleanup is performed before next step
            shard.ClearSensorCache(fullRefreshInterval, minInterCleanupInterval,
                stepStartTime + minInterCleanupInterval - TDuration::Seconds(1));
            UNIT_ASSERT_VALUES_EQUAL(size, shard.SensorCacheSize());
        }
        for (const auto& key: allKeys) {
            auto sensorId = shard.FindSensor(key);
            UNIT_ASSERT(sensorId);
            // check that there are no sensor ids left that were added at step 0
            UNIT_ASSERT(sensorId->ShardId > 0);
        }
    }

    Y_UNIT_TEST(LookupWithTypeCheck) {
        TMetabaseShardKey shardKey;
        TMetabaseShard shard(shardKey);
        auto key = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=prestable"), TSignalName(TString("signal1_hgram")));
        auto sensor = TSensorId(0, 0, MetricType::LOG_HISTOGRAM);
        shard.SaveSensors({{key, sensor}});

        TSensorTypeMap table1({
            {yandex::solomon::model::MetricType::LOG_HISTOGRAM, yandex::solomon::model::MetricType::HIST}
        });
        UNIT_ASSERT_EQUAL(sensor, shard.FindSensorWithTypeCheck(TTypedSeriesKey{key, MetricType::LOG_HISTOGRAM}, table1));
        UNIT_ASSERT(!shard.FindSensorWithTypeCheck(TTypedSeriesKey{key, MetricType::HIST}, table1));
        TSensorTypeMap table2({
            {yandex::solomon::model::MetricType::HIST, yandex::solomon::model::MetricType::LOG_HISTOGRAM}
        });
        UNIT_ASSERT_EQUAL(sensor, shard.FindSensorWithTypeCheck(TTypedSeriesKey{key, MetricType::LOG_HISTOGRAM}, table2));
        UNIT_ASSERT_EQUAL(sensor, shard.FindSensorWithTypeCheck(TTypedSeriesKey{key, MetricType::HIST}, table2));
    }

    Y_UNIT_TEST(SaveRejected) {
        TMetabaseShardKey shardKey;
        TMetabaseShard shard(shardKey, TInstant::Seconds(100));
        auto key1 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=prestable"), TSignalName(TString("signal1_summ")));
        auto key2 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=unstable"), TSignalName(TString("signal2_summ")));

        auto now = TInstant::Now();
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key1, now));
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key2, now));
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key2, now));

        shard.SaveRejectedSensors({{key1, now + TDuration::Minutes(5)}});

        UNIT_ASSERT_EQUAL(true, shard.ContainsInRejection(key1, now));
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key2, now));
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key1, now + TDuration::Minutes(10)));
    }

    Y_UNIT_TEST(ClearRejectedByTime) {
        TMetabaseShardKey shardKey;
        TMetabaseShard shard(shardKey, TInstant::Seconds(100));
        auto key1 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=prestable"), TSignalName(TString("signal1_summ")));
        auto key2 = TSeriesKey::Make(TInstanceKey::FromNamed("yasmtestit|ctype=unstable"), TSignalName(TString("signal2_summ")));

        auto now = TInstant::Now();
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key1, now));
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key2, now));

        shard.SaveRejectedSensors({{key1, now + TDuration::Minutes(5)}});
        shard.SaveRejectedSensors({{key2, now + TDuration::Minutes(5)}});

        UNIT_ASSERT_EQUAL(true, shard.ContainsInRejection(key1, now));
        UNIT_ASSERT_EQUAL(true, shard.ContainsInRejection(key2, now));

        shard.ClearSensorCache(TDuration::Seconds(200), TDuration::Seconds(1), now + TDuration::Minutes(10));

        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key1, now));
        UNIT_ASSERT_EQUAL(false, shard.ContainsInRejection(key2, now));
    }
}
