#include "points.h"
#include "metrics.h"

#include <infra/yasm/common/points/hgram/hgram.h>

#include <solomon/protos/stockpile/stockpile_requests.pb.h>
#include <solomon/protos/metabase/metric.pb.h>

#include <util/generic/xrange.h>
#include <util/generic/ymath.h>

using namespace NHistDb::NStockpile;
using namespace NHistDb::NStockpile::NImpl;
using namespace NZoom::NValue;
using namespace NZoom::NHgram;
using namespace NZoom::NAccumulators;
using NZoom::NValue::EValueType;
using yandex::solomon::model::MetricType;

namespace {
    static constexpr TStockpileColumnMask UGRAM_COLUMN_MASK{
            yandex::solomon::stockpile::EColumnFlag::COLUMN_TIMESTAMP |
            yandex::solomon::stockpile::EColumnFlag::COLUMN_HISTOGRAM};
    static constexpr TStockpileColumnMask LOG_HGRAM_COLUMN_MASK{
            yandex::solomon::stockpile::EColumnFlag::COLUMN_TIMESTAMP |
            yandex::solomon::stockpile::EColumnFlag::COLUMN_LOG_HISTOGRAM};
    static constexpr TStockpileColumnMask DSUMMARY_COLUMN_MASK{
            yandex::solomon::stockpile::EColumnFlag::COLUMN_TIMESTAMP |
            yandex::solomon::stockpile::EColumnFlag::COLUMN_DSUMMARY};

    static constexpr NProtoBuf::uint32 LOG_HGRAM_MAX_BUCKET_SIZE = 100;
    static constexpr double LOG_HGRAM_BASE = 1.5;

    void CheckType(MetricType metabaseType, MetricType dataType) {
        if (metabaseType != dataType) {
            ythrow TTypeNotSupportedException()
                << "metabase_type=" << metabaseType << ", actual_type=" << dataType;
        }
    }

    struct THgramVisitor: public IHgramStorageCallback {
    public:
        THgramVisitor(TPoint& point, TRecordSerializeState& recordSerializeState)
            : Point(point)
            , RecordSerializeState(recordSerializeState)
            , IsEmpty(false)
        {}

        void OnStoreSmall(const TVector<double>& values, const size_t zeros) override {
            auto type = RecordSerializeState.GetType();
            switch (type) {
                case MetricType::HIST: {
                    TUgram ugram{};
                    ugram.MulSmallHgram(values, zeros);
                    OnStoreHistogram(ugram.GetBuckets());
                    break;
                }
                case MetricType::LOG_HISTOGRAM: {
                    // NOTE(rocco66): force cast to log hgram
                    TNormal normal{TVector<double>(), 0, -1};
                    normal.MulSmallHgram(values, zeros);
                    OnStoreLogHistogram(normal.GetBuckets(), normal.GetZerosCount(), normal.GetStartPower());
                    break;
                }
                default: {
                    ythrow TTypeNotSupportedException()
                        << "metabase_type=" << type << ", actual_type=" << MetricType::LOG_HISTOGRAM;
                }
            }
        }

        void OnStoreNormal(const TVector<double>& values, const size_t zeros, const i16 startPower) override {
            auto type = RecordSerializeState.GetType();
            switch (type) {
                case MetricType::HIST: {
                    TUgram ugram{};
                    ugram.MulNormalHgram(values, zeros, startPower);
                    OnStoreHistogram(ugram.GetBuckets());
                    break;
                }
                case MetricType::LOG_HISTOGRAM: {
                    OnStoreLogHistogram(values, zeros, startPower);
                    break;
                }
                default: {
                    ythrow TTypeNotSupportedException()
                        << "metabase_type=" << type << ", actual_type=" << MetricType::LOG_HISTOGRAM;
                }
            }
        }

        void OnStoreUgram(const TUgramBuckets& buckets) override {
            auto type = RecordSerializeState.GetType();
            switch (type) {
                case MetricType::HIST: {
                    OnStoreHistogram(buckets);
                    break;
                }
                case MetricType::LOG_HISTOGRAM: {
                    if (buckets.empty()) {
                        IsEmpty = true;
                        break;
                    }

                    THolder<IHgramImpl> normalUgram = THgramNormal::FromUgram(buckets);
                    normalUgram->Store(*this);
                    break;
                }
                default: {
                    ythrow TTypeNotSupportedException()
                            << "metabase_type=" << type << ", actual_type=" << MetricType::HIST;
                }
            }
        }

        bool Empty() const {
            return IsEmpty;
        };
    private:
        void OnStoreLogHistogram(const TVector<double>& values, const size_t zeros, const i16 startPower) {
            auto grpcLogHgram = Point.MutableLogHistogram();
            grpcLogHgram->set_zeroes(zeros);
            grpcLogHgram->set_max_buckets_size(LOG_HGRAM_MAX_BUCKET_SIZE);
            grpcLogHgram->set_base(LOG_HGRAM_BASE);

            size_t from = 0;
            while (from < values.size() && values[from] == 0) {
                ++from;
            }

            // no points
            if (from >= values.size()) {
                grpcLogHgram->set_start_power(0);
                return;
            }

            size_t to = values.size();
            while (to > 0 && values[to - 1] == 0) {
                --to;
            }

            grpcLogHgram->set_start_power(static_cast<i32>(startPower) + static_cast<i32>(from));
            for (auto index = from; index < to; ++index) {
                grpcLogHgram->add_buckets(values[index]);
            }
        }

        void OnStoreHistogram(const TUgramBuckets& buckets) {
            if (buckets.empty()) {
                IsEmpty = true;
                return;
            }

            auto& bucketsWithFixedBounds = RecordSerializeState.FixUgramBuckets(buckets);
            ::yandex::solomon::model::Histogram* grpcHgram = Point.MutableHistogram();

            size_t processedBucketsCounter{};
            const TUgramBucket* prevBucket = nullptr;
            for (const auto& bucket : bucketsWithFixedBounds) {
                processedBucketsCounter++;
                if (prevBucket == nullptr || prevBucket->UpperBound != bucket.LowerBound) {
                    grpcHgram->add_bounds(bucket.LowerBound);
                    grpcHgram->add_buckets(0);
                    if (static_cast<size_t>(grpcHgram->buckets_size()) >= STOCKPILE_HISTOGRAM_BUCKETS_LIMIT - 1) {
                        // TODO(rocco66): remove trim after https://st.yandex-team.ru/GOLOVAN-6252
                        break;
                    }
                }
                prevBucket = &bucket;
                auto bound =  bucket.UpperBound;
                if (bucket.IsPoint()) {
                    bound = StockpilePoint(*grpcHgram->bounds().rbegin());
                }

                grpcHgram->add_bounds(bound);
                grpcHgram->add_buckets(round(bucket.Weight));
                if (static_cast<size_t>(grpcHgram->buckets_size()) >= STOCKPILE_HISTOGRAM_BUCKETS_LIMIT - 1) {
                    // TODO(rocco66): remove trim after https://st.yandex-team.ru/GOLOVAN-6252
                    break;
                }
            }

            // NOTE(rocco66): see https://st.yandex-team.ru/GOLOVAN-6544
            grpcHgram->add_bounds(LAST_INF_UGRAM_BOARD);
            grpcHgram->add_buckets(0);

            auto droppedBuckets = (bucketsWithFixedBounds.size() - processedBucketsCounter);
            if (droppedBuckets > 0) {
                RecordSerializeState.IncCroppedUgramCounter();
                TUnistat::Instance().PushSignalUnsafe(NMetrics::DROPPED_UGRAM_BUCKETS, droppedBuckets);
            }
        }

    private:
        TPoint& Point;
        TRecordSerializeState& RecordSerializeState;
        bool IsEmpty;
    };

    class TSummarySummDeserializer final: public IValueDeserializerImpl {
    public:
        TValue Decode(const yandex::solomon::stockpile::TPoint& point) override {
            double value = point.GetSummaryDouble().sum();
            return IsNan(value) ? TValue() : TValue(value);
        }
    };

    class TCountedSumDeserializer final: public IValueDeserializerImpl {
    public:
        TValue Decode(const yandex::solomon::stockpile::TPoint& point) override {
            const auto& summary(point.GetSummaryDouble());
            return TValue(summary.sum(), summary.count());
        }
    };

    class TSummaryMinDeserializer final: public IValueDeserializerImpl {
    public:
        TValue Decode(const yandex::solomon::stockpile::TPoint& point) override {
            double value = point.GetSummaryDouble().min();
            return IsNan(value) ? TValue() : TValue(value);
        }
    };

    class TSummaryMaxDeserializer final: public IValueDeserializerImpl {
    public:
        TValue Decode(const yandex::solomon::stockpile::TPoint& point) override {
            double value = point.GetSummaryDouble().max();
            return IsNan(value) ? TValue() : TValue(value);
        }
    };

    class TSummaryLastDeserializer final: public IValueDeserializerImpl {
    public:
        TValue Decode(const yandex::solomon::stockpile::TPoint& point) override {
            double value = point.GetSummaryDouble().last();
            return IsNan(value) ? TValue() : TValue(value);
        }
    };

    class TLogHgramDeserializer final: public IValueDeserializerImpl {
    public:
        TValue Decode(const yandex::solomon::stockpile::TPoint& point) override {
            const auto& hgram(point.GetLogHistogram());
            TVector<double> buckets;
            buckets.reserve(hgram.buckets_size());
            buckets.insert(buckets.end(), hgram.buckets().begin(), hgram.buckets().end());
            return TValue(THgram::Normal(std::move(buckets), hgram.zeroes(), hgram.start_power()));
        }
    };

    class TUgramDeserializer final: public IValueDeserializerImpl {
    public:
        TValue Decode(const yandex::solomon::stockpile::TPoint& point) override {
            TUgramBuckets buckets;
            const auto& hgram(point.GetHistogram());
            if (hgram.bounds_size() != 0) {
                buckets.reserve(hgram.bounds_size());
                double lastBound = hgram.bounds(0);
                if (hgram.buckets(0) > 0) {
                    // NOTE(rocco66): we have right board only, so let it be point GOLOVAN-6617
                    buckets.emplace_back(TUgramBucket::Point(lastBound, hgram.buckets(0)));
                }
                for (const auto idx : xrange(1, hgram.bounds_size())) {
                    if (hgram.bounds(idx) == lastBound) {
                        buckets.emplace_back(TUgramBucket::Point(hgram.bounds(idx), hgram.buckets(idx)));
                    } else if (hgram.buckets(idx) > 0) {
                        buckets.emplace_back(TUgramBucket(lastBound, hgram.bounds(idx), hgram.buckets(idx)));
                    }
                    lastBound = hgram.bounds(idx);
                }
            }
            return TValue(THgram::Ugram(std::move(buckets)));
        }
    };

    THolder<IValueDeserializerImpl> CreateDeserializer(MetricType type, EAccumulatorType aggregationType) {
        switch (type) {
            case MetricType::LOG_HISTOGRAM: {
                return MakeHolder<TLogHgramDeserializer>();
            }
            case MetricType::HIST: {
                return MakeHolder<TUgramDeserializer>();
            }
            case MetricType::DSUMMARY: {
                switch (aggregationType) {
                    case EAccumulatorType::Min: {
                        return MakeHolder<TSummaryMinDeserializer>();
                    }
                    case EAccumulatorType::Summ:
                    case EAccumulatorType::SummNone: {
                        return MakeHolder<TSummarySummDeserializer>();
                    }
                    case EAccumulatorType::Max: {
                        return MakeHolder<TSummaryMaxDeserializer>();
                    }
                    case EAccumulatorType::Last: {
                        return MakeHolder<TSummaryLastDeserializer>();
                    }
                    case EAccumulatorType::Avg:
                    case EAccumulatorType::Average: {
                        return MakeHolder<TCountedSumDeserializer>();
                    }
                    default: {
                        ythrow TTypeNotSupportedException() << "unsupported sensor aggregation type: " << aggregationType;
                    }
                }
            }
            default: {
                ythrow TTypeNotSupportedException() << "unsupported sensor type: " << type;
            }
        }
    }
}

MetricType NHistDb::NStockpile::ToSensorType(EValueType valueType, EAccumulatorType aggregationType) {
    switch (valueType) {
        case EValueType::NONE: {
            return MetricType::METRIC_TYPE_UNSPECIFIED;
        }
        case EValueType::HYPER_LOGLOG: {
            return MetricType::METRIC_TYPE_UNSPECIFIED;
        }
        case EValueType::COUNTED_SUM: {
            return MetricType::DSUMMARY;
        }
        case EValueType::FLOAT: {
            switch (aggregationType) {
                case EAccumulatorType::Summ:
                case EAccumulatorType::SummNone:
                case EAccumulatorType::Average:
                case EAccumulatorType::Min:
                case EAccumulatorType::Max:
                case EAccumulatorType::Last: {
                    return MetricType::DSUMMARY;
                }
                default: {
                    return MetricType::METRIC_TYPE_UNSPECIFIED;
                }
            }
        }
        case EValueType::NORMAL_HGRAM: {
            return MetricType::LOG_HISTOGRAM;
        }
        case EValueType::SMALL_HGRAM: {
            return MetricType::LOG_HISTOGRAM;
        }
        case EValueType::USER_HGRAM: {
            return MetricType::HIST;
        }
        case EValueType::VEC: {
            return MetricType::METRIC_TYPE_UNSPECIFIED;
        }
    }
}

EValueType NHistDb::NStockpile::FromSensorType(MetricType type, EAccumulatorType aggregationType) {
    switch (type) {
        case MetricType::LOG_HISTOGRAM: {
            return EValueType::NORMAL_HGRAM;
        }
        case MetricType::HIST: {
            return EValueType::USER_HGRAM;
        }
        case MetricType::DSUMMARY: {
            switch (aggregationType) {
                case EAccumulatorType::Avg:
                case EAccumulatorType::Average: {
                    return EValueType::COUNTED_SUM;
                }
                case EAccumulatorType::Min:
                case EAccumulatorType::Max:
                case EAccumulatorType::Summ:
                case EAccumulatorType::SummNone:
                case EAccumulatorType::Last: {
                    return EValueType::FLOAT;
                }
                default: {
                    return EValueType::NONE;
                }
            }
        }
        default: {
            return EValueType::NONE;
        }
    }
}

TStockpileColumnMask NHistDb::NStockpile::ToColumnMask(MetricType type) {
    switch (type) {
        case MetricType::LOG_HISTOGRAM: {
            return LOG_HGRAM_COLUMN_MASK;
        }
        case MetricType::HIST: {
            return UGRAM_COLUMN_MASK;
        }
        case MetricType::DSUMMARY: {
            return DSUMMARY_COLUMN_MASK;
        }
        default: {
            ythrow TTypeNotSupportedException() << type << " is not supported";
        }
    }
}

EAccumulatorType NHistDb::NStockpile::GetAggregationType(const NZoom::NSignal::TSignalName& signal, EAggregationMethod method) {
    return signal.GetAggregationType(method).GetRef();
}

TValueTypeDetector::TValueTypeDetector(EAccumulatorType aggregationType)
    : ValueType(NZoom::NValue::EValueType::NONE)
    , AggregationType(aggregationType)
{
}

void TValueTypeDetector::OnValue(TValueRef value) {
    const EValueType given = FromSensorType(ToSensorType(value.GetType(), AggregationType), AggregationType);
    if (given == EValueType::NONE) {
        return;
    }
    if (ValueType == EValueType::NONE) {
        ValueType = given;
    } else if (given != ValueType) {
        // emulate standard hgram evolution from normal to ugram
        if (ValueType == EValueType::NORMAL_HGRAM && given == EValueType::USER_HGRAM) {
            ValueType = EValueType::USER_HGRAM;
        } else if (ValueType == EValueType::USER_HGRAM && given == EValueType::NORMAL_HGRAM) {
            ValueType = EValueType::USER_HGRAM;
        } else {
            ythrow TTypeNotSupportedException() << "can't convert " << given << " to " << ValueType;
        }
    }
}

void TValueSerializer::MulNone() {
    IsEmpty = true;
}

void TValueSerializer::MulFloat(const double value) {
    switch (RecordSerializeState.GetAggregationType()) {
        case EAccumulatorType::Summ:
        case EAccumulatorType::SummNone: {
            CheckType(GetType(), MetricType::DSUMMARY);
            Point.MutableSummaryDouble()->set_sum(value);
            break;
        }
        case EAccumulatorType::Min: {
            CheckType(GetType(), MetricType::DSUMMARY);
            Point.MutableSummaryDouble()->set_min(value);
            break;
        }
        case EAccumulatorType::Max: {
            CheckType(GetType(), MetricType::DSUMMARY);
            Point.MutableSummaryDouble()->set_max(value);
            break;
        }
        case EAccumulatorType::Last: {
            CheckType(GetType(), MetricType::DSUMMARY);
            Point.MutableSummaryDouble()->set_last(value);
            break;
        }
        case EAccumulatorType::Average: {
            CheckType(GetType(), MetricType::DSUMMARY);
            Point.MutableSummaryDouble()->set_sum(value);
            Point.MutableSummaryDouble()->set_count(1);
            break;
        }
        default: {
            IsEmpty = true;
            IsError = true;
            return;
        }
    }
    Point.SetTimestampsMillis(Timestamp.MilliSeconds());
}

void TValueSerializer::MulVec(const TVector<double>&) {
    IsEmpty = true;
}

void TValueSerializer::MulCountedSum(const double sum, const ui64 count) {
    CheckType(GetType(), MetricType::DSUMMARY);
    Point.MutableSummaryDouble()->set_sum(sum);
    Point.MutableSummaryDouble()->set_count(count);
    Point.SetTimestampsMillis(Timestamp.MilliSeconds());
}

void TValueSerializer::MulHgram(const THgram& value) {
    THgramVisitor serializer(Point, RecordSerializeState);
    value.Store(serializer);
    if (serializer.Empty()) {
        IsEmpty = true;
    } else {
        Point.SetTimestampsMillis(Timestamp.MilliSeconds());
    }
}

bool TValueSerializer::Empty() const {
    return IsEmpty;
}

bool TValueSerializer::Error() const {
    return IsError;
}

MetricType TValueSerializer::GetType() const {
    return RecordSerializeState.GetType();
}

TStockpileColumnMask TValueSerializer::GetColumnMask() const {
    return ToColumnMask(GetType());
}

TValueDeserializer::TValueDeserializer(MetricType type, EAccumulatorType aggregationType)
    : Deserializer(CreateDeserializer(type, aggregationType))
{
}

TValue TValueDeserializer::Decode(const yandex::solomon::stockpile::TPoint& point) {
    return Deserializer->Decode(point);
}
