#pragma once

#include "record_serialize_state.h"

#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/stockpile_client/common/base_types.h>

#include <solomon/protos/metabase/grpc_status.pb.h>
#include <solomon/protos/stockpile/stockpile_service.grpc.pb.h>

#include <util/datetime/base.h>

namespace NHistDb::NStockpile {
    static constexpr double LAST_INF_UGRAM_BOARD = std::numeric_limits<double>::max();

    namespace NImpl {
        class IValueDeserializerImpl {
        public:
            virtual ~IValueDeserializerImpl() = default;
            virtual NZoom::NValue::TValue Decode(const yandex::solomon::stockpile::TPoint& point) = 0;
        };
    }

    using TStockpileColumnMask = NProtoBuf::uint32;
    using TPoint = ::yandex::solomon::stockpile::TPoint;

    class TTypeNotSupportedException: public yexception {
    };

    yandex::solomon::model::MetricType ToSensorType(
        NZoom::NValue::EValueType valueType,
        NZoom::NAccumulators::EAccumulatorType aggregationType);
    NZoom::NValue::EValueType FromSensorType(
        yandex::solomon::model::MetricType type,
        NZoom::NAccumulators::EAccumulatorType aggregationType);
    TStockpileColumnMask ToColumnMask(yandex::solomon::model::MetricType type);
    NZoom::NAccumulators::EAccumulatorType GetAggregationType(
            const NZoom::NSignal::TSignalName& signal,
            NZoom::NAccumulators::EAggregationMethod method = NZoom::NAccumulators::EAggregationMethod::Rollup
                    );

    class TValueTypeDetector {
    public:
        TValueTypeDetector(NZoom::NAccumulators::EAccumulatorType aggregationType);

        void OnValue(NZoom::NValue::TValueRef value);

        yandex::solomon::model::MetricType GetType() const noexcept {
            return ToSensorType(ValueType, AggregationType);
        }

    private:
        NZoom::NValue::EValueType ValueType;
        NZoom::NAccumulators::EAccumulatorType AggregationType;
    };

    inline double StockpilePoint(double d) {
        return std::nextafter(d, LAST_INF_UGRAM_BOARD);
    }

    class TValueSerializer final: public NZoom::NValue::IUpdatable {
    public:
        TValueSerializer(TPoint& point, TInstant timestamp, TRecordSerializeState& recordSerializeState)
            : Point(point)
            , Timestamp(timestamp)
            , RecordSerializeState(recordSerializeState)
            , IsEmpty(false)
            , IsError(false)
        {
        }

        void MulNone() override;
        void MulFloat(double value) override;
        void MulVec(const TVector<double>& value) override;
        void MulCountedSum(double sum, ui64 count) override;
        void MulHgram(const NZoom::NHgram::THgram& value) override;

        bool Empty() const;
        bool Error() const;

        yandex::solomon::model::MetricType GetType() const;
        TStockpileColumnMask GetColumnMask() const;

    private:
        yandex::solomon::stockpile::TPoint& Point;
        TInstant Timestamp;
        TRecordSerializeState& RecordSerializeState;
        bool IsEmpty;
        bool IsError;
    };

    class TValueDeserializer {
    public:
        TValueDeserializer(yandex::solomon::model::MetricType type, NZoom::NAccumulators::EAccumulatorType aggregationType);

        NZoom::NValue::TValue Decode(const yandex::solomon::stockpile::TPoint& point);

    private:
        THolder<NImpl::IValueDeserializerImpl> Deserializer;
    };
}
