#include "points.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NHistDb::NStockpile;
using namespace NZoom::NValue;
using namespace NZoom::NHgram;
using namespace NZoom::NAccumulators;
using namespace yandex::solomon::metabase;
using yandex::solomon::model::MetricType;

namespace {
    static constexpr TInstant TIMESTAMP = TInstant::Days(100);

    TSensorId MakeSensorId(MetricType type) {
        return TSensorId{0, 0, type};
    }

    TValue EncodeDecodeHelper(TValueRef incoming, TSensorId sensorId, EAccumulatorType aggregationType) {
        TPoint point;
        auto state = TRecordSerializeState{sensorId, 42, aggregationType};
        TValueSerializer serializer(point, TIMESTAMP, state);
        incoming.Update(serializer);
        UNIT_ASSERT(!serializer.Empty());
        UNIT_ASSERT_VALUES_EQUAL(TInstant::MilliSeconds(point.GetTimestampsMillis()), TIMESTAMP);
        TValueDeserializer deserializer(sensorId.Type, aggregationType);
        return deserializer.Decode(point);
    }

    template<class RawVector>
    TVector<double> ConvertVector(const RawVector& vec) {
        return {vec.begin(), vec.end()};
    }
}

Y_UNIT_TEST_SUITE(TGrpcDeserialize) {

    Y_UNIT_TEST(UgramDeserializer) {
        TValueDeserializer deserializer {
            yandex::solomon::model::MetricType::HIST,
            NZoom::NAccumulators::EAccumulatorType::Hgram
        };
        yandex::solomon::stockpile::TPoint point{};
        auto* hgram = point.MutableHistogram();
        hgram->add_bounds(10);
        hgram->add_bounds(20);
        hgram->add_bounds(LAST_INF_UGRAM_BOARD);
        hgram->add_buckets(1);
        hgram->add_buckets(2);
        hgram->add_buckets(3);
        NZoom::NValue::TValue parsedValue {deserializer.Decode(point)};
        NZoom::NValue::TValue oracleValue {NZoom::NHgram::THgram::Ugram({
            NZoom::NHgram::TUgramBucket{10, 10, 1},
            NZoom::NHgram::TUgramBucket{10, 20, 2},
            NZoom::NHgram::TUgramBucket{20, LAST_INF_UGRAM_BOARD, 3},
        })};
        UNIT_ASSERT_VALUES_EQUAL(parsedValue, oracleValue);
    }
}

Y_UNIT_TEST_SUITE(TPointsTest) {
    void CheckSerialize(
            TValue input,
            MetricType type,
            EAccumulatorType aggregationType,
            TMaybe<TValue> expected = Nothing()
        ) {
        TValue answer(EncodeDecodeHelper(input.GetValue(), MakeSensorId(type), aggregationType));
        UNIT_ASSERT_VALUES_EQUAL(expected.GetOrElse(input), answer);
    }

    Y_UNIT_TEST(None) {
        TPoint point;
        auto state = TRecordSerializeState{MakeSensorId(MetricType::DSUMMARY), 42,EAccumulatorType::Summ};
        TValueSerializer serializer(point, TIMESTAMP, state);
        TValue().Update(serializer);
        UNIT_ASSERT(serializer.Empty());
    }

    Y_UNIT_TEST(Double) {
        CheckSerialize(TValue(42.0), MetricType::DSUMMARY, EAccumulatorType::Summ);
    }

    Y_UNIT_TEST(DoubleNan) {
        TValue given(std::nan(""));
        TValue answer(EncodeDecodeHelper(given.GetValue(), MakeSensorId(MetricType::DSUMMARY), EAccumulatorType::Summ));
        UNIT_ASSERT_VALUES_EQUAL(answer, TValue());
    }

    Y_UNIT_TEST(CountedSum) {
        CheckSerialize(TValue(42.0, 15), MetricType::DSUMMARY, EAccumulatorType::Average);
    }

    Y_UNIT_TEST(LogHgram) {
        CheckSerialize(TValue(THgram::Normal(TVector<double>{1, 2, 3, 4}, 15, 1)), MetricType::LOG_HISTOGRAM, EAccumulatorType::Hgram);
    }

    Y_UNIT_TEST(EmptyUgram) {
        TPoint point;
        auto state = TRecordSerializeState{MakeSensorId(MetricType::HIST), 42,EAccumulatorType::Hgram};
        TValueSerializer serializer(point, TIMESTAMP, state);
        TValue(THgram::EmptyUgram()).Update(serializer);
        UNIT_ASSERT(serializer.Empty());
    }

    Y_UNIT_TEST(EmptyUgramInResponse) {
        TPoint point;
        TValueDeserializer deserializer(MetricType::HIST, EAccumulatorType::Hgram);
        auto hgram(deserializer.Decode(point));
        UNIT_ASSERT_VALUES_EQUAL(hgram, TValue(THgram::EmptyUgram()));
    }

    Y_UNIT_TEST(Ugram) {
        CheckSerialize(
            TValue(THgram::Ugram(TUgramBuckets{
                TUgramBucket{1.0, 2.0, 10},
                TUgramBucket{2.0, 3.0, 20},
                TUgramBucket{4.0, 4.0, 30},
                TUgramBucket{4.0, 5.0, 40},
                TUgramBucket{5.0, 5.0, 0},
            })),
            MetricType::HIST,
            EAccumulatorType::Hgram,
            {
                TValue(THgram::Ugram(TUgramBuckets{
                    TUgramBucket{1.0, 2.0, 10},
                    TUgramBucket{2.0, 3.0, 20},
                    TUgramBucket{4.0, StockpilePoint(4.0), 30},
                    TUgramBucket{StockpilePoint(4.0), 5.0, 40},
                })),
            });
    }

        Y_UNIT_TEST(UgramRoundingWeights) {
        CheckSerialize(
            TValue(THgram::Ugram(TUgramBuckets{
                TUgramBucket{5.0, 6.0, 10.1},
                TUgramBucket{6.0, 7.0, 10.9},
                TUgramBucket{7.0, 8.0, 10.5},
                TUgramBucket{8.0, 9.0, 1.0},
                TUgramBucket{9.0, 10.0, 0.3},
                TUgramBucket{10.0, 11.0, 0.8},
            })),
            MetricType::HIST,
            EAccumulatorType::Hgram,
            {
                TValue(THgram::Ugram(TUgramBuckets{
                    TUgramBucket{5.0, 6.0, 10},
                    TUgramBucket{6.0, 7.0, 11},
                    TUgramBucket{7.0, 8.0, 11},
                    TUgramBucket{8.0, 9.0, 1},
                    TUgramBucket{10.0, 11.0, 1},
                })),
            });
    }

    Y_UNIT_TEST(UgramFreezeLogic) {
        class TMockedRecord final: public NZoom::NRecord::IRecordIterable {
        public:
            void Iterate(NZoom::NRecord::IRecordVisitor& visitor) const override {
                // NOTE(rocco66): we need at least two different ugrams for freeze
                auto firstValue = TValue(THgram::Ugram(TUgramBuckets{
                    TUgramBucket{1.0, 3.0, 1}
                }));
                visitor.OnValue(firstValue.GetValue());
                auto secondValue = TValue(THgram::Ugram(TUgramBuckets{
                    TUgramBucket{4.0, 5.0, 1}
                }));
                visitor.OnValue(secondValue.GetValue());
            }
        };

        auto origin = TValue(THgram::Ugram(TUgramBuckets{
            TUgramBucket{1.0, 2.0, 10},
            TUgramBucket{2.0, 3.0, 20},
            TUgramBucket{4.0, 4.0, 30},
            TUgramBucket{4.0, 5.0, 40},
            TUgramBucket{5.0, 5.0, 0},
        }));
        auto sensorId = MakeSensorId(MetricType::HIST);
        auto aggregationType = EAccumulatorType::Hgram;

        TPoint point;
        TMockedRecord record;
        TRecordSerializeState state {sensorId, 42, aggregationType};

        TRecordUgramMergerVisitor ugramVisitor;
        record.Iterate(ugramVisitor);
        state.SetUgramFreezer(ugramVisitor.GetFreezer());

        TValueSerializer serializer(point, TIMESTAMP, state);
        origin.GetValue().Update(serializer);
        TValueDeserializer deserializer(sensorId.Type, aggregationType);
        TValue answer {deserializer.Decode(point)};

        auto expected = TValue(THgram::Ugram(TUgramBuckets{
            TUgramBucket{1.0, 3.0, 30},
            TUgramBucket{4.0, 5.0, 70},
        }));
        UNIT_ASSERT_VALUES_EQUAL(expected, answer);
    }
}

Y_UNIT_TEST_SUITE(UgramRawBucketsLimit) {
    class TMockedRecord final: public NZoom::NRecord::IRecordIterable {
    public:
        TMockedRecord(const TUgramBuckets& buckets)
            : Buckets(buckets)
        {
        }

        void Iterate(NZoom::NRecord::IRecordVisitor& visitor) const override {
            // NOTE(rocco66): we need at least two different ugrams for freeze
            auto firstValue = TValue(THgram::Ugram({Buckets[0]}));
            visitor.OnValue(firstValue.GetValue());
            auto secondValue = TValue(THgram::Ugram(std::move(Buckets)));
            visitor.OnValue(secondValue.GetValue());
        }
    private:
        mutable TUgramBuckets Buckets;
    };

    class TMockedValueRecord final: public NZoom::NRecord::IRecordIterable {
    public:
        TMockedValueRecord(TValue& values)
            : Value(values)
        {
        }

        void Iterate(NZoom::NRecord::IRecordVisitor& visitor) const override {
            visitor.OnValue(Value.GetValue());
        }
    private:
        TValue& Value;
    };

    using TBounds = TVector<double>;
    using TWeights = TVector<double>;

    void Test(
            const TUgramBuckets& buckets,
            const size_t limit,
            const TVector<double> resultBounds,
            const TVector<double> resultBuckets
    ) {
        auto bucketsCopy = buckets;
        auto origin = TValue(THgram::Ugram(std::move(bucketsCopy)));
        auto sensorId = MakeSensorId(MetricType::HIST);
        auto aggregationType = EAccumulatorType::Hgram;

        TPoint point;
        TMockedRecord record {buckets};
        TRecordSerializeState state {sensorId, 42, aggregationType};

        TRecordUgramMergerVisitor ugramVisitor;
        record.Iterate(ugramVisitor);
        state.SetUgramFreezer(ugramVisitor.GetFreezer(limit));

        TValueSerializer serializer(point, TIMESTAMP, state);
        origin.GetValue().Update(serializer);
        TValueDeserializer deserializer(sensorId.Type, aggregationType);
        TValue answer {deserializer.Decode(point)};

        auto bounds = ConvertVector(point.GetHistogram().bounds());
        UNIT_ASSERT(bounds.size() <= limit);
        UNIT_ASSERT_VALUES_EQUAL(bounds, resultBounds);
        auto rawBuckets = ConvertVector(point.GetHistogram().buckets());
        UNIT_ASSERT(rawBuckets.size() <= limit);
        UNIT_ASSERT_VALUES_EQUAL(rawBuckets, resultBuckets);
    }

    Y_UNIT_TEST(NoCompress) {
        Test(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
                TUgramBucket(3.0, 4.0, 1.0),
            },
            6,
            TBounds{1., 2., 3., 4., LAST_INF_UGRAM_BOARD},
            TWeights{0., 1., 1., 1., 0.}
        );
    }

    Y_UNIT_TEST(CompressOneRawBucket) {
        Test(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
                TUgramBucket(3.0, 4.0, 1.0),
            },
            4,
            TBounds{1., 2., 4., LAST_INF_UGRAM_BOARD},
            TWeights{0., 1., 2., 0}
        );
    }

    Y_UNIT_TEST(Points) {
        // NOTE(rocco66): point is two raw buckets
        Test(
            {
                TUgramBucket(1.0, 1.0, 1.0),
                TUgramBucket(2.0, 2.0, 1.0),
                TUgramBucket(3.0, 3.0, 1.0),
            },
            6,
            TBounds{1., StockpilePoint(1.), 2., 3., LAST_INF_UGRAM_BOARD},
            TWeights{0., 1., 0., 2, 0}
        );
    }

    Y_UNIT_TEST(PointAtLowerBorder) {
        Test(
            {
                TUgramBucket(1.0, 1.0, 1.0),
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
            },
            4,
            TBounds{1., StockpilePoint(1.), 3., LAST_INF_UGRAM_BOARD},
            TWeights{0., 1., 2., 0.}
        );
    }

    Y_UNIT_TEST(PointAtUpperBorder) {
        Test(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
                TUgramBucket(3.0, 3.0, 1.0),
            },
            4,
            TBounds{1., 3., StockpilePoint(3.), LAST_INF_UGRAM_BOARD},
            TWeights{0., 2., 1.,  0}
        );
    }

    Y_UNIT_TEST(PointBetweenTwoIntervals) {
        Test(
            {
                TUgramBucket(1.0, 2.0, 1.0),
                TUgramBucket(2.0, 2.0, 1.0),
                TUgramBucket(2.0, 3.0, 1.0),
            },
            4,
            TBounds{1., 2., 3., LAST_INF_UGRAM_BOARD},
            TWeights{0., 1., 2., 0}
        );
    }

    Y_UNIT_TEST(WrongCompressionExample) {
        Test(
            {
                TUgramBucket(0, 0.1, 472),
                TUgramBucket(0.1, 0.2, 107),
                TUgramBucket(0.2, 0.3, 75),
                TUgramBucket(0.3, 0.4, 49),
                TUgramBucket(0.4, 0.5, 77),
                TUgramBucket(0.5, 0.6, 64),
                TUgramBucket(0.6, 0.7, 70),
                TUgramBucket(0.7, 0.8, 79),
                TUgramBucket(0.8, 0.9, 81),
                TUgramBucket(0.9, 0.905, 4),
                TUgramBucket(0.905, 0.91, 2),
                TUgramBucket(0.91, 0.915, 2),
                TUgramBucket(0.915, 0.92, 7),
                TUgramBucket(0.92, 0.925, 9),
                TUgramBucket(0.925, 0.93, 8),
                TUgramBucket(0.93, 0.935,14),
                TUgramBucket(0.935, 0.94, 7),
                TUgramBucket(0.94, 0.945, 11),
                TUgramBucket(0.945, 0.95, 17),
                TUgramBucket(0.95, 0.955, 6),
                TUgramBucket(0.955, 0.9575, 10),
                TUgramBucket(0.9575, 0.96, 8),
                TUgramBucket(0.96, 0.9625, 10),
                TUgramBucket(0.9625, 0.965, 15),
                TUgramBucket(0.965, 0.9675, 10),
                TUgramBucket(0.9675, 0.97, 8),
                TUgramBucket(0.97, 0.9725, 25),
                TUgramBucket(0.9725, 0.975, 20),
                TUgramBucket(0.975, 0.9775, 21),
                TUgramBucket(0.9775, 0.98, 22),
                TUgramBucket(0.98, 0.98125, 9),
                TUgramBucket(0.98125, 0.9825, 7),
                TUgramBucket(0.9825, 0.98375, 11),
                TUgramBucket(0.98375, 0.985, 12),
                TUgramBucket(0.985, 0.98625, 11),
                TUgramBucket(0.98625, 0.9875, 10),
                TUgramBucket(0.9875, 0.98875, 13),
                TUgramBucket(0.98875, 0.99,21),
                TUgramBucket(0.99, 0.99125, 11),
                TUgramBucket(0.99125, 0.9925, 15),
                TUgramBucket(0.9925, 0.99375, 31),
                TUgramBucket(0.99375, 0.995, 25),
                TUgramBucket(0.995, 0.99625, 23),
                TUgramBucket(0.99625, 0.9975, 32),
                TUgramBucket(0.9975, 0.99875, 15),
                TUgramBucket(0.99875, 0.999, 1),
                TUgramBucket(0.999, 0.9999, 13),
                TUgramBucket(0.9999, 1, 4),
                TUgramBucket(1, 1, 468)
            },
            50,
            TBounds{0, 0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 0.905, 0.915, 0.92, 0.925, 0.93, 0.935, 0.94, 0.945, 0.95, 0.955, 0.9575, 0.96, 0.9625, 0.965, 0.9675, 0.97, 0.9725, 0.975, 0.9775, 0.98, 0.98125, 0.9825, 0.98375, 0.985, 0.98625, 0.9875, 0.98875, 0.99, 0.99125, 0.9925, 0.99375, 0.995, 0.99625, 0.9975, 0.99875, 0.999, 0.9999, 1, StockpilePoint(1), LAST_INF_UGRAM_BOARD},
            TWeights{0, 472, 107, 75, 49, 77, 64, 70, 79, 81, 4, 4, 7, 9, 8, 14, 7, 11, 17, 6, 10, 8, 10, 15, 10, 8, 25, 20, 21, 22, 9, 7, 11, 12, 11, 10, 13, 21, 11, 15, 31, 25, 23, 32, 15, 1, 13, 4, 468, 0.}
        );
    }

    Y_UNIT_TEST(SmallHgramCompressionExample) {
        // GOLOVAN-6252
        auto sensorId = MakeSensorId(MetricType::HIST);
        auto aggregationType = EAccumulatorType::Hgram;

        const int limit = 7;

        TPoint point;

        TValue smallHgram {THgram::Small(xrange(10), 1)};
        TValue value {smallHgram.GetValue()};
        TMockedValueRecord record {value};
        TLog logger{};
        TRecordSerializeState state {sensorId, 42, aggregationType};

        TRecordUgramMergerVisitor ugramVisitor;
        record.Iterate(ugramVisitor);
        state.SetUgramFreezer(ugramVisitor.GetFreezer(limit));

        TValueSerializer serializer(point, TIMESTAMP, state);
        smallHgram.GetValue().Update(serializer);
        TValueDeserializer deserializer(sensorId.Type, aggregationType);
        TValue answer {deserializer.Decode(point)};

        auto bounds = ConvertVector(point.GetHistogram().bounds());
        UNIT_ASSERT(bounds.size() <= limit);
        TVector<double> resultBounds {
            0,
            StockpilePoint(0),
            1,
            4,
            5,
            9,
            LAST_INF_UGRAM_BOARD
        };
        UNIT_ASSERT_VALUES_EQUAL(bounds, resultBounds);
        auto rawBuckets = ConvertVector(point.GetHistogram().buckets());
        UNIT_ASSERT(rawBuckets.size() <= limit);
        TVector<double> resultBuckets {
            0,
            2,
            0,
            4,
            0,
            5,
            0
        };
        UNIT_ASSERT_VALUES_EQUAL(rawBuckets, resultBuckets);
    }

    Y_UNIT_TEST(LogHgramToUgramCompressionExample) {
        auto sensorId = MakeSensorId(MetricType::HIST);
        auto aggregationType = EAccumulatorType::Hgram;

        const int limit = 8;

        TPoint point;

        TValue normalHgram {THgram::Normal(xrange(10), 1, 0)};
        TValue value {normalHgram.GetValue()};
        TMockedValueRecord record {value};
        TLog logger{};
        TRecordSerializeState state {sensorId, 42, aggregationType};

        TRecordUgramMergerVisitor ugramVisitor;
        record.Iterate(ugramVisitor);
        state.SetUgramFreezer(ugramVisitor.GetFreezer(limit));

        TValueSerializer serializer(point, TIMESTAMP, state);
        normalHgram.GetValue().Update(serializer);
        TValueDeserializer deserializer(sensorId.Type, aggregationType);
        TValue answer {deserializer.Decode(point)};

        auto bounds = point.GetHistogram().bounds();
        UNIT_ASSERT(bounds.size() <= limit);
        TVector<double> resultBounds {
            0,
            StockpilePoint(0),
            std::pow(1.5, 1),
            std::pow(1.5, 5),
            std::pow(1.5, 7),
            std::pow(1.5, 9),
            std::pow(1.5, 10),
            LAST_INF_UGRAM_BOARD
        };
        UNIT_ASSERT_VALUES_EQUAL(ConvertVector(bounds), resultBounds);
        auto rawBuckets = ConvertVector(point.GetHistogram().buckets());
        UNIT_ASSERT(rawBuckets.size() <= limit);
        TVector<double> resultBuckets {
            0,
            1,
            0,
            10,
            11,
            15,
            9,
            0
        };
        UNIT_ASSERT_VALUES_EQUAL(rawBuckets, resultBuckets);
    }

    Y_UNIT_TEST(InteravalAndPoint) {
        TVector<TValue> values;
        values.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{3.0, 3.0, 10}}));
        values.emplace_back(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 2.0, 10}}));

        TRecordUgramMergerVisitor ugramVisitor;
        for (const auto& val : values) {
            ugramVisitor.OnValue(val.GetValue());
        }

        auto sensorId = MakeSensorId(MetricType::HIST);
        auto aggregationType = EAccumulatorType::Hgram;
        TRecordSerializeState state {sensorId, 42, aggregationType};
        state.SetUgramFreezer(ugramVisitor.GetFreezer());

        TVector<TPoint> points;
        for (const auto& val : values) {
            points.emplace_back();
            TValueSerializer serializer(points.back(), TIMESTAMP, state);
            val.GetValue().Update(serializer);
        }

        auto bounds0 = ConvertVector(points[0].GetHistogram().bounds());
        for (const auto& point : points) {
            auto bounds = ConvertVector(point.GetHistogram().bounds());
            UNIT_ASSERT_VALUES_EQUAL(bounds, bounds0);
        }
    }
}

Y_UNIT_TEST_SUITE(TValueTypeDetectorTest) {
    Y_UNIT_TEST(DefaultValue) {
        TValueTypeDetector detector(EAccumulatorType::Hgram);
        UNIT_ASSERT_VALUES_EQUAL(MetricType::METRIC_TYPE_UNSPECIFIED, detector.GetType());
    }

    Y_UNIT_TEST(DifferenHgramValues) {
        TValueTypeDetector detector1(EAccumulatorType::Hgram);
        detector1.OnValue(TValue(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 3.0, 10}})).GetValue());
        detector1.OnValue(TValue(THgram::Default(true)).GetValue());
        UNIT_ASSERT_VALUES_EQUAL(MetricType::HIST, detector1.GetType());

        TValueTypeDetector detector2(EAccumulatorType::Hgram);
        detector2.OnValue(TValue(THgram::Default(true)).GetValue());
        detector2.OnValue(TValue(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 3.0, 10}})).GetValue());
        UNIT_ASSERT_VALUES_EQUAL(MetricType::HIST, detector2.GetType());

        TValueTypeDetector detector3(EAccumulatorType::Hgram);
        detector3.OnValue(TValue(THgram::Default(false)).GetValue());
        detector3.OnValue(TValue(THgram::Ugram(TUgramBuckets{TUgramBucket{1.0, 3.0, 10}})).GetValue());
        UNIT_ASSERT_VALUES_EQUAL(MetricType::HIST, detector3.GetType());

        TValueTypeDetector detector4(EAccumulatorType::Hgram);
        detector4.OnValue(TValue(THgram::Default(false)).GetValue());
        detector4.OnValue(TValue(THgram::Default(true)).GetValue());
        UNIT_ASSERT_VALUES_EQUAL(MetricType::LOG_HISTOGRAM, detector4.GetType());
    }
}
