#include "rpc.h"

#include <util/string/cast.h>
#include <util/network/ip.h>

#include <contrib/libs/grpc/src/core/lib/iomgr/socket_mutator.h>
#include <contrib/libs/grpc/src/core/lib/gpr/useful.h>

// NOTE(rocco66): have not idea why #include <netinet/tcp.h> does not work
#ifndef TCP_USER_TIMEOUT
#define TCP_USER_TIMEOUT 0x12
#endif

using namespace NHistDb::NStockpile;

namespace {
    constexpr size_t MAX_RECEIVE_MESSAGE = 128 * 1024 * 1024;
    constexpr size_t MAX_SEND_MESSAGE = 128 * 1024 * 1024;
    constexpr int USER_MUTATOR_TIMEOUT_MS = 20000;

    class TUserTimeoutMutator : public grpc_socket_mutator {
    public:
        TUserTimeoutMutator(TLog* log, int timeoutMs)
            : Log(log)
            , TimeoutMs(timeoutMs)
        {
            grpc_socket_mutator_init(this, &Vtable);
        }

    private:
        static bool FdMutate(int fd, grpc_socket_mutator* mutator) {
            auto self = Cast(mutator);
            // NOTE(rocco66): partly from https://github.com/grpc/grpc/pull/16419/files#diff-947502d9f646d02105a2988588866c01R295
            int newval;
            socklen_t len = sizeof(newval);
            if (0 != setsockopt(fd, IPPROTO_TCP, TCP_USER_TIMEOUT, &self->TimeoutMs, sizeof(self->TimeoutMs))) {
                if (self->Log)
                    *self->Log << TLOG_ERR << "setsockopt(TCP_USER_TIMEOUT): " << errno;
                return false;
            }
            if (0 != getsockopt(fd, IPPROTO_TCP, TCP_USER_TIMEOUT, &newval, &len)) {
                if (self->Log)
                    *self->Log << TLOG_ERR << "getsockopt(TCP_USER_TIMEOUT): " << errno;
                return false;
            }
            if (newval != self->TimeoutMs) {
                if (self->Log)
                    *self->Log << TLOG_ERR << "Failed to set TCP_USER_TIMEOUT (" << newval << " vs " << self->TimeoutMs << "%s)";
                return false;
            }
            return true;
        }

        static int Compare(grpc_socket_mutator* a, grpc_socket_mutator* b) {
            return grpc_core::QsortCompare(a, b);
        }

        static void Destroy(grpc_socket_mutator* mutator) {
            delete mutator;
        }

        static TUserTimeoutMutator* Cast(grpc_socket_mutator* mutator) {
            return static_cast<TUserTimeoutMutator*>(mutator);
        }

        constexpr static grpc_socket_mutator_vtable Vtable{
            .mutate_fd = TUserTimeoutMutator::FdMutate,
            .compare = TUserTimeoutMutator::Compare,
            .destroy = TUserTimeoutMutator::Destroy
        };

        TLog* Log;
        int TimeoutMs;
    };
} // namespace

TGrpcSettings::TGrpcSettings(const TString& clientName, TLog* log, bool useKeepAlive)
    : Log(log)
{
    Arguments.SetMaxSendMessageSize(MAX_SEND_MESSAGE);
    Arguments.SetMaxReceiveMessageSize(MAX_RECEIVE_MESSAGE);
    Arguments.SetCompressionAlgorithm(GRPC_COMPRESS_GZIP);
    Arguments.SetUserAgentPrefix(YASM_USER_AGENT);
    Arguments.SetInt(GRPC_ARG_INITIAL_RECONNECT_BACKOFF_MS, 1000);
    Arguments.SetInt(GRPC_ARG_MAX_RECONNECT_BACKOFF_MS, 10000);

    if (useKeepAlive) {
        Arguments.SetInt(GRPC_ARG_KEEPALIVE_TIME_MS, 5000);
        Arguments.SetInt(GRPC_ARG_KEEPALIVE_TIMEOUT_MS, 2000);
    } else {
        // Mutator logic was initially used because of https://github.com/grpc/grpc/issues/15889
        // Kept now for cases when keep alive is not needed.
#ifdef _linux_
        // NOTE(rocco66): mutator will be destroyed inside grpc
        auto* mutator = new TUserTimeoutMutator(Log, USER_MUTATOR_TIMEOUT_MS);
        Arguments.SetSocketMutator(mutator);
#else
        Arguments.SetInt(GRPC_ARG_KEEPALIVE_TIME_MS, USER_MUTATOR_TIMEOUT_MS);
        Arguments.SetInt(GRPC_ARG_KEEPALIVE_TIMEOUT_MS, USER_MUTATOR_TIMEOUT_MS);
#endif
    }


    TStringBuilder clientId;
    clientId << clientName;
    auto tags = NTags::TInstanceKey::FromRtcEnviron().GetTags();
    for (auto& tag: tags) {
        if (tag.Name == "ctype") {
            clientId << "_" << tag.Value;
            break;
        }
    }
    SolomonClientId = clientId;

    gpr_set_log_function(&TGrpcSettings::LogGrpcMessageToSingletonLog);
}

void TGrpcSettings::LogGrpcMessageToSingletonLog(gpr_log_func_args* args) {
    TGrpcSettings::Get().LogGrpcMessage(args);
}

void TGrpcSettings::LogGrpcMessage(gpr_log_func_args* args) {
    if (Log) {
        *Log << ELogPriority::TLOG_INFO << "[grpc-log] (" << gpr_log_severity_string(args->severity) << ") "
             << args->file << ':' << args->line << " " << args->message;
    }
}

ui64 NHistDb::NStockpile::ToDeadlineMillis(TDuration timeout) {
    return timeout.ToDeadLine().MilliSeconds();
}

TGrpcState::TGrpcState()
    : ExecutionStatus(IN_PROGRESS)
{
}

void TGrpcState::SetStartTime(TInstant now) {
    StartTime = now;
}

void TGrpcState::SaveTimingMetric() {
    Duration = TInstant::Now() - StartTime;
}

void TGrpcState::PushTimingMetric() {
    TUnistat::Instance().PushSignalUnsafe(GetRequestName(), Duration.SecondsFloat());

    grpc::ClientContext* context(GetContext());
    if (context != nullptr) {
        auto it = context->GetServerInitialMetadata().find("x-solomon-created-at");
        if (it != context->GetServerInitialMetadata().end()) {
            auto metricName = NMetrics::TStockpileStatsInitializer::MakeRttMetricName(GetRequestName());
            try {
                TStringBuf value(it->second.data(), it->second.size());
                auto roundTripTime(TInstant::Now() - TInstant::MilliSeconds(FromString<i64>(value)));
                TUnistat::Instance().PushSignalUnsafe(metricName, roundTripTime.SecondsFloat());
            } catch(...) {
            }
        }
    }
}

void TGrpcState::PushFailMetric() {
    auto metricName = NMetrics::TStockpileStatsInitializer::MakeFailMetricName(GetRequestName());
    TUnistat::Instance().PushSignalUnsafe(metricName, 1);
}

void TGrpcState::MarkAs(EExecutionStatus status) {
    ExecutionStatus = status;
    Y_ASSERT(IsFinished());
    ExecuteCallback();
}

bool TGrpcState::IsFinished() const {
    switch (ExecutionStatus) {
        case IN_PROGRESS:
            return false;
        case FINISHED:
        case TERMINAL_FAILURE:
        case RETRIABLE_FAILURE:
        case TOPOLOGY_FAILURE:
            return true;
    }
}

bool TGrpcState::IsFailed() const {
    switch (ExecutionStatus) {
        case IN_PROGRESS:
        case FINISHED:
            return false;
        case TERMINAL_FAILURE:
        case RETRIABLE_FAILURE:
        case TOPOLOGY_FAILURE:
            return true;
    }
}

bool TGrpcState::IsRetriable() const {
    switch (ExecutionStatus) {
        case RETRIABLE_FAILURE:
        case TOPOLOGY_FAILURE:
            return true;
        default:
            return false;
    }
}

bool TGrpcState::TopologyChanged() const {
    return ExecutionStatus == TOPOLOGY_FAILURE;
}

bool TGrpcState::IsSuccess() const {
    return ExecutionStatus == FINISHED;
}

grpc::ClientContext* TGrpcState::GetContext() {
    return nullptr;
}

void TGrpcState::Cancel() {
    grpc::ClientContext* context(GetContext());
    if (context != nullptr) {
        context->TryCancel();
    }
}

TGrpcCompletionQueue::TGrpcCompletionQueue()
    : CallsInFlight(0)
{
}

TGrpcCompletionQueue::~TGrpcCompletionQueue() {
    Queue.Shutdown();
    void* state = nullptr;
    bool ok = false;
    while (Queue.Next(&state, &ok)) {
    }
}

void TGrpcCompletionQueue::Wait(IGrpcStateHandler& stateHandler) {
    while (CallsInFlight && WaitAsync(stateHandler, TDuration::MilliSeconds(10))) {
    }
    Y_VERIFY(!CallsInFlight);
}

bool TGrpcCompletionQueue::WaitAsync(IGrpcStateHandler& stateHandler) {
    return WaitAsync(stateHandler, TDuration::Zero());
}

bool TGrpcCompletionQueue::WaitAsync(IGrpcStateHandler& stateHandler, TDuration timeout) {
    const auto deadline = std::chrono::system_clock::now() + std::chrono::milliseconds(timeout.MilliSeconds());
    bool gotEvent = false;
    while (CallsInFlight) {
        void* state = nullptr;
        bool ok = false;
        switch (Queue.AsyncNext(&state, &ok, deadline)) {
            case grpc::CompletionQueue::TIMEOUT: {
                return true;
            }
            case grpc::CompletionQueue::GOT_EVENT: {
                CallsInFlight--;
                stateHandler.Handle(*static_cast<TGrpcState*>(state), ok);
                gotEvent = true;
                break;
            }
            case grpc::CompletionQueue::SHUTDOWN: {
                return gotEvent;
            }
        }
    }
    return gotEvent;
}

template <>
void Out<TGrpcRemoteHost>(IOutputStream& stream,
                          TTypeTraits<TGrpcRemoteHost>::TFuncParam remoteHost) {
    stream << "TGrpcRemoteHost{.HostName=\"" << remoteHost.GetHost() << "\", .Port=" << remoteHost.GetPort() << ", .Database=" << remoteHost.GetDatabase() << "}";
}
