#pragma once

#include <infra/yasm/stockpile_client/common/base_types.h>
#include "metrics.h"

#include <solomon/protos/metabase/grpc_status.pb.h>
#include <solomon/protos/metabase/metabase_service.grpc.pb.h>
#include <solomon/protos/stockpile/status_code.pb.h>
#include <solomon/protos/stockpile/stockpile_service.grpc.pb.h>
#include <solomon/services/dataproxy/api/dataproxy_service.pb.h>
#include <solomon/services/dataproxy/api/dataproxy_service.grpc.pb.h>

#include <library/cpp/unistat/unistat.h>
#include <library/cpp/logger/log.h>
#include <util/datetime/base.h>
#include <util/system/type_name.h>
#include <util/generic/maybe.h>

#include <contrib/libs/grpc/include/grpcpp/grpcpp.h>

namespace NHistDb::NStockpile {
    using TMetabaseService = yandex::monitoring::metabase::MetabaseService;
    using TStockpileService = yandex::monitoring::stockpile::StockpileService;
    using TDataProxyService = yandex::monitoring::dataproxy::DataProxyService;

    class TGrpcSettings {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        static constexpr bool USE_KEEP_ALIVE_BY_DEFAULT = false;

        static void Init(const TString& clientName, TLog& log, bool useKeepAlive = USE_KEEP_ALIVE_BY_DEFAULT) {
            SingletonWithPriority<TGrpcSettings, 100001>(clientName, &log, (bool)useKeepAlive);
        }

        static TGrpcSettings& Get() {
            static const TString DEFAULT_CLIENT_NAME = "yasmdefault";
            return *SingletonWithPriority<TGrpcSettings, 100001>(
                DEFAULT_CLIENT_NAME,
                (TLog*)nullptr,
                (bool)USE_KEEP_ALIVE_BY_DEFAULT);
        }

        const grpc::ChannelArguments& GetChannelArguments() const { return Arguments; }
        inline TString GetSolomonClientId() const { return SolomonClientId; }

    private:
        TGrpcSettings(const TString& clientName, TLog* log, bool useKeepAlive);
        static void LogGrpcMessageToSingletonLog(gpr_log_func_args* args);
        void LogGrpcMessage(gpr_log_func_args* args);

        TString SolomonClientId;
        grpc::ChannelArguments Arguments;
        TLog* Log;
    };

    ui64 ToDeadlineMillis(TDuration timeout);

    class TGrpcRemoteHost {
    public:
        TGrpcRemoteHost(TString hostname, unsigned short port, std::shared_ptr<grpc::Channel> channel)
            : Hostname(std::move(hostname))
            , Port(port)
            , Channel(std::move(channel))
        {
        }

        virtual ~TGrpcRemoteHost() = default;

        const TString& GetHost() const {
            return Hostname;
        }

        unsigned short GetPort() const {
            return Port;
        };

        std::shared_ptr<grpc::Channel> GetChannel() const {
            return Channel;
        };

        virtual EStockpileDatabase GetDatabase() const = 0;

        bool operator==(const TGrpcRemoteHost& other) const noexcept {
            return Hostname == other.Hostname && Port == other.Port;
        }

        bool operator!=(const TGrpcRemoteHost& other) const noexcept {
            return !(*this == other);
        }

    private:
        const TString Hostname;
        const unsigned short Port;
        std::shared_ptr<grpc::Channel> Channel;
    };

    class TMetabaseRemoteHost final: public TGrpcRemoteHost {
    public:
        TMetabaseRemoteHost(TString hostname, unsigned short port, std::shared_ptr<grpc::Channel> channel)
            : TGrpcRemoteHost(std::move(hostname), port, std::move(channel))
            , Stub(TMetabaseService::NewStub(GetChannel()))
        {
        }

        TMetabaseService::Stub* GetStub() const {
            return Stub.get();
        }

        EStockpileDatabase GetDatabase() const override {
            return EStockpileDatabase::Metabase;
        }

    private:
        std::unique_ptr<TMetabaseService::Stub> Stub;
    };

    class TStockpileRemoteHost final: public TGrpcRemoteHost {
    public:
        TStockpileRemoteHost(TString hostname, unsigned short port, std::shared_ptr<grpc::Channel> channel)
            : TGrpcRemoteHost(std::move(hostname), port, std::move(channel))
            , Stub(TStockpileService::NewStub(GetChannel()))
        {
        }

        TStockpileService::Stub* GetStub() const {
            return Stub.get();
        }

        EStockpileDatabase GetDatabase() const override {
            return EStockpileDatabase::Stockpile;
        }

    private:
        std::unique_ptr<TStockpileService::Stub> Stub;
    };

    class TDataProxyRemoteHost final: public TGrpcRemoteHost {
    public:
        TDataProxyRemoteHost(TString hostname, unsigned short port, std::shared_ptr<grpc::Channel> channel)
            : TGrpcRemoteHost(std::move(hostname), port, std::move(channel))
            , Stub(TDataProxyService::NewStub(GetChannel()))
        {
        }

        TDataProxyService::Stub* GetStub() const {
            return Stub.get();
        }

        EStockpileDatabase GetDatabase() const override {
            return EStockpileDatabase::DataProxy;
        }

    private:
        std::unique_ptr<TDataProxyService::Stub> Stub;
    };

    class TGrpcState {
    public:
        enum EExecutionStatus {
            IN_PROGRESS,
            FINISHED,
            TERMINAL_FAILURE,
            RETRIABLE_FAILURE,
            TOPOLOGY_FAILURE
        };

        TGrpcState();

        virtual ~TGrpcState() = default;

        void SetStartTime(TInstant now);
        void SaveTimingMetric();
        void PushTimingMetric();
        void PushFailMetric();

        virtual void MarkAs(EExecutionStatus status);

        bool IsFinished() const;
        bool IsFailed() const;
        bool IsRetriable() const;
        bool TopologyChanged() const;
        bool IsSuccess() const;
        TDuration GetDuration() const {
            return Duration;
        }

        virtual void Handle() = 0;

        virtual TStringBuf GetRequestName() const = 0;

        virtual grpc::ClientContext* GetContext();

        virtual void Cancel();

        template <class F>
        void SetCallback(F&& callback) {
            Callback = std::move(callback);
        }

    protected:
        void ExecuteCallback() {
            if (Callback) {
                Callback(*this);
            }
        }

        TInstant StartTime;
        TDuration Duration;
        EExecutionStatus ExecutionStatus;

        std::function<void(TGrpcState&)> Callback;
    };

    template <class TStub>
    struct TGrpcRemoteHostSelector;

    template <>
    struct TGrpcRemoteHostSelector<TStockpileService::Stub> {
        using TType = TStockpileRemoteHost;
    };

    template <>
    struct TGrpcRemoteHostSelector<TMetabaseService::Stub> {
        using TType = TMetabaseRemoteHost;
    };

    template <>
    struct TGrpcRemoteHostSelector<TDataProxyService::Stub> {
        using TType = TDataProxyRemoteHost;
    };

    template <class F>
    class TGrpcAsyncCallState;

    class TRetriableFailure : public yexception {};
    class TTopologyFailure : public yexception {};

    enum class EGrpcRetryModeFlags {
        // individual bits
        ON_DEADLINE         = 1, // Retry on timeouts
        ON_INTERNAL_ERROR   = 2  // Retry internal server errors
    };

    Y_DECLARE_FLAGS(TGrpcRetryMode, EGrpcRetryModeFlags);
    Y_DECLARE_OPERATORS_FOR_FLAGS(TGrpcRetryMode);

    template <class TStub, class TRequest, class TResponse>
    class TGrpcAsyncCallState<std::unique_ptr<grpc::ClientAsyncResponseReader<TResponse>> (TStub::*)(grpc::ClientContext*, TRequest&, grpc::CompletionQueue*)> {
    public:
        using TRequestType = std::decay_t<TRequest>;
        using TResponseType = std::decay_t<TResponse>;
        using TMethod = std::unique_ptr<grpc::ClientAsyncResponseReader<TResponse>> (TStub::*)(grpc::ClientContext*, TRequest&, grpc::CompletionQueue*);
        using TRemoteHost = typename TGrpcRemoteHostSelector<TStub>::TType;

        TGrpcAsyncCallState(TMethod method, TAtomicSharedPtr<TGrpcRemoteHost> remoteHost, TGrpcRetryMode retryMode,
            bool treatNotFoundAsError = true)
            : Method(method)
            , RemoteHost(std::move(remoteHost))
            , RetryMode(retryMode)
            , TreatNotFoundAsError(treatNotFoundAsError)
            , State(MakeHolder<TState>())
        {
        }

        TGrpcAsyncCallState(TGrpcAsyncCallState&& other)
            : Method(std::move(other.Method))
            , RemoteHost(std::move(other.RemoteHost))
            , RetryMode(other.RetryMode)
            , TreatNotFoundAsError(other.TreatNotFoundAsError)
            , State(std::move(other.State))
            , ResponseReader(std::move(other.ResponseReader))
        {
        }

        TGrpcAsyncCallState RecreateWithNewHost(TAtomicSharedPtr<TGrpcRemoteHost> remoteHost) {
            auto state = TGrpcAsyncCallState(Method, std::move(remoteHost), RetryMode, TreatNotFoundAsError);
            state.GetRequest().CopyFrom(GetRequest());
            return state;
        }

        grpc::ClientContext& GetContext() noexcept {
            return State->Context;
        }

        TRequestType& GetRequest() noexcept {
            return *State->Request;
        }

        TResponseType& GetResponse() {
            Check();
            return *State->Response;
        }

        const TRemoteHost& GetRemoteHost() const {
            return dynamic_cast<const TRemoteHost&>(*RemoteHost);
        }

        auto& Execute(TGrpcState& state, grpc::CompletionQueue& completionQueue) {
            TInstant now = TInstant::Now();
            state.SetStartTime(now);

            TString created(ToString(now.MilliSeconds()));
            State->Context.AddMetadata("x-solomon-clientid", TGrpcSettings::Get().GetSolomonClientId());
            State->Context.AddMetadata("x-solomon-created-at", created);

            ResponseReader = (GetRemoteHost().GetStub()->*Method)(&State->Context, *State->Request, &completionQueue);
            ResponseReader->StartCall();
            ResponseReader->Finish(State->Response, &State->Status, &state);

            return *this;
        }

        auto& Check() const {
            if (!State->Status.ok()) {
                auto errorCode = State->Status.error_code();
                if (errorCode != grpc::StatusCode::NOT_FOUND || TreatNotFoundAsError) {
                    TString message = TStringBuilder() << "Invalid " << TypeName<TResponse>() << " grpc response from "
                                                       << *RemoteHost << " (" << errorCode << "): "
                                                       << State->Status.error_message();
                    switch (errorCode) {
                        case grpc::StatusCode::UNAVAILABLE: {
                            ythrow TRetriableFailure() << message;
                        }
                        case grpc::StatusCode::DEADLINE_EXCEEDED: {
                            if (RetryMode.HasFlags(EGrpcRetryModeFlags::ON_DEADLINE)) {
                                ythrow TRetriableFailure() << message;
                            } else {
                                ythrow yexception() << message;
                            }
                        }
                        case grpc::StatusCode::INTERNAL:
                        case grpc::StatusCode::RESOURCE_EXHAUSTED: {
                            if (RetryMode.HasFlags(EGrpcRetryModeFlags::ON_INTERNAL_ERROR)) {
                                ythrow TRetriableFailure() << message;
                            } else {
                                ythrow yexception() << message;
                            }
                        }
                        default: {
                            ythrow yexception() << message;
                        }
                    }
                }
            }
            return *this;
        }

        auto& SetDeadline(TDuration timeout) {
            const auto deadline = std::chrono::system_clock::now() + std::chrono::milliseconds(timeout.MilliSeconds());
            State->Context.set_deadline(deadline);
            return *this;
        }

    private:
        struct TState {
            TState()
                : Request(::google::protobuf::Arena::CreateMessage<TRequestType>(&Arena))
                , Response(::google::protobuf::Arena::CreateMessage<TResponseType>(&Arena))
            {
            }

            ::google::protobuf::Arena Arena;
            grpc::ClientContext Context;
            TRequestType* Request = nullptr;
            TResponseType* Response = nullptr;
            grpc::Status Status;
        };

        TMethod Method;
        TAtomicSharedPtr<TGrpcRemoteHost> RemoteHost;
        const TGrpcRetryMode RetryMode;
        const bool TreatNotFoundAsError;
        THolder<TState> State;
        std::unique_ptr<grpc::ClientAsyncResponseReader<TResponse>> ResponseReader;
    };

    template<typename TMethod>
    auto GrpcAsyncCallState(TMethod method, TAtomicSharedPtr<TGrpcRemoteHost> remoteHost, TGrpcRetryMode retryMode,
                            bool treatNotFoundAsError = true) -> TGrpcAsyncCallState<decltype(method)> {
        TGrpcAsyncCallState<decltype(method)> state(method, std::move(remoteHost), retryMode, treatNotFoundAsError);
        return std::move(state);
    }

    class IGrpcStateHandler {
    public:
        virtual ~IGrpcStateHandler() = default;

        virtual void Handle(TGrpcState& state, bool ok) noexcept = 0;
    };

    class TGrpcCompletionQueue {
    public:
        TGrpcCompletionQueue();
        ~TGrpcCompletionQueue();

        void Wait(IGrpcStateHandler& stateHandler);
        bool WaitAsync(IGrpcStateHandler& stateHandler);
        bool WaitAsync(IGrpcStateHandler& stateHandler, TDuration timeout);
        size_t GetCallsInFlight() const {
            return CallsInFlight;
        }

        template <class TCall>
        TCall& Execute(TCall& asyncCall, TGrpcState& state) {
            CallsInFlight++;
            return asyncCall.Execute(state, Queue);
        }

    private:
        size_t CallsInFlight;
        grpc::CompletionQueue Queue;
    };

    template <class TLogger>
    class TGrpcStateLoggingHandler final: public IGrpcStateHandler {
    public:
        TGrpcStateLoggingHandler(TLogger& log)
            : Queue()
            , Log(log)
        {
        }

        void Handle(TGrpcState& state, bool ok) noexcept override  {
            Y_VERIFY(ok);
            state.SaveTimingMetric();
            try {
                state.Handle();
                state.MarkAs(TGrpcState::FINISHED);
            } catch (const TTopologyFailure&) {
                Log << TLOG_WARNING << "Topology failure: " << CurrentExceptionMessage();
                state.MarkAs(TGrpcState::TOPOLOGY_FAILURE);
            } catch (const TRetriableFailure&) {
                Log << TLOG_WARNING << "Retriable failure: " << CurrentExceptionMessage();
                state.MarkAs(TGrpcState::RETRIABLE_FAILURE);
            } catch (...) {
                Log << TLOG_ERR << "Unknown handler error: " << CurrentExceptionMessage();
                state.MarkAs(TGrpcState::TERMINAL_FAILURE);
            }
            if (state.IsFailed()) {
                state.PushFailMetric();
            }
            state.PushTimingMetric();
        }

        void Wait() {
            Queue.Wait(*this);
        }

        bool WaitAsync() {
            return Queue.WaitAsync(*this);
        }

        bool WaitAsync(TDuration timeout) {
            return Queue.WaitAsync(*this, timeout);
        }

        TGrpcCompletionQueue& GetQueue() {
            return Queue;
        }

        size_t GetCallsInFlight() const {
            return Queue.GetCallsInFlight();
        }

        template <class TCall>
        TCall& Execute(TCall& asyncCall, TGrpcState& state) {
            return Queue.Execute(asyncCall, state);
        }

    private:
        TGrpcCompletionQueue Queue;
        TLogger& Log;
    };

    using TGrpcStateHandler = TGrpcStateLoggingHandler<TLog>;
}
