#pragma once

#include <infra/yasm/common/config/fast_config.h>

#include <library/cpp/config/config.h>
#include <library/cpp/getopt/last_getopt.h>

#include <util/folder/path.h>
#include <util/generic/fwd.h>
#include <util/system/env.h>
#include <util/stream/file.h>

namespace NHistDb::NStockpile {
    enum class EStockpileClusterType {
        Production,
        Testing,
        Prestable,
    };

    class TWrongClusterType: public yexception {};

    inline EStockpileClusterType GetStockpileClusterType(const TString& clusterTypeString) {
        if (clusterTypeString == "production") {
            return EStockpileClusterType::Production;
        } else if (clusterTypeString == "testing") {
            return EStockpileClusterType::Testing;
        } else if (clusterTypeString == "prestable") {
            return EStockpileClusterType::Prestable;
        } else {
            ythrow TWrongClusterType() << clusterTypeString;
        }
    }

    struct TClusterInfo {
        TClusterInfo(EStockpileClusterType clusterType, const TString& clusterName)
            : ClusterType(clusterType)
            , ClusterName(clusterName)
        {}

        EStockpileClusterType ClusterType;
        TString ClusterName;
    };

    using TShardIndex = ui64;

    class TMetabaseConfig {
    public:
        TMetabaseConfig(TStringBuf filePath);
        TShardIndex GetShardsCount(TStringBuf itype) const;
    private:
        THashMap<TString, TShardIndex> Config;
    };

    using TFastConfigSettings = NYasm::NCommon::TFastConfigSettings;
    using TFastConfig = NYasm::NCommon::TFastConfig;

    struct TSettings {
        TClusterInfo ClusterInfo;
        TString AuthToken;
        bool CreateNewShards;
        TMetabaseConfig MetabaseShardsConfig;
    };

    class TStockpileSetingsFactory {
    public:
        NStockpile::TSettings CreateStockpileSettings(bool createNewShards) const;

        static void FillParser(NLastGetopt::TOpts& options);
        void ProcessParsedOptions(const NLastGetopt::TOptsParseResult& parsed);
        void FillFromFileConfig(const NConfig::TConfig& config);

    protected:

        inline void SetStockpileClusterTypeFromString(const TString& value) {
            StockpileClusterType = GetStockpileClusterType(value);
        }

        inline void SetStockpileClusterName(const TString& value) {
            StockpileClusterName = value;
        }

        NStockpile::TClusterInfo GetClusterInfo() const;

        inline void SetAuthToken(const TString& value) {
            AuthToken = value;
        }

        inline void SetStockpileMetabaseConfigPath(const TString &value) {
            StockpileMetabaseConfigPath = value;
        }
    private:
        TString AuthToken;
        TMaybe<EStockpileClusterType> StockpileClusterType;
        TString StockpileClusterName;
        TFsPath StockpileMetabaseConfigPath;
    };
}
