#pragma once

#include "proto_to_slog_converter.h"

#include <solomon/libs/cpp/timeseries/timeseries.h>
#include <solomon/libs/cpp/slog/slog.h>
#include <solomon/libs/cpp/slog/log_data.h>
#include <solomon/libs/cpp/slog/resolved_meta/builder.h>

namespace NHistDb::NStockpile {

struct TMeta {
public:
    TMeta() = default;
    explicit TMeta(TNumId numId) {
        Encoder = NSolomon::NSlog::NResolvedMeta::CreateResolvedMetaBuilder(
            numId,
            yandex::solomon::stockpile::EDecimPolicy::POLICY_5_MIN_AFTER_8_DAYS,
            0,
            0,
            NMonitoring::ECompression::LZ4,
            &Out);
        ++Metrics;
    }

    void Finish() {
        Encoder->Close();
        Out.Finish();
    }

public:
    TString Content{};
    TStringOutput Out{Content};
    NSolomon::NSlog::NResolvedMeta::IResolvedMetaBuilderPtr Encoder;
    size_t Metrics{0};
};

struct TData {
public:
    TData() = default;
    explicit TData(TNumId numId) {
        Encoder = NSolomon::NSlog::CreateLogDataBuilder(
            numId,
            NSolomon::NSlog::EDataCodingScheme::Slog,
            NMonitoring::ETimePrecision::SECONDS,
            NMonitoring::ECompression::LZ4,
            &Out);
    }

    ui32 WritePoints(NSolomon::NTsModel::EPointType pointType, const NSolomon::TAggrTimeSeries& series) {
        ++Metrics;
        Points += series.Size();
        return Encoder->OnTimeSeries(pointType, static_cast<NSolomon::ELogFlagsComb>(NSolomon::ELogFlags::None), series);
    }

    void Finish() {
        Encoder->Close();
        Out.Finish();
    }

public:
    TString Content{};
    TStringOutput Out{Content};
    NSolomon::NSlog::ILogDataBuilderPtr Encoder;
    bool HasCommonTime{false};
    size_t Metrics{0};
    size_t Points{0};
};

struct TSlogResult {
    TString Index;
    TString Content;
};

class TSlogBuilder {
public:
    TSlogBuilder() = default;

    void OnMetricBegin(TNumId numId, TLocalId localId, yandex::solomon::model::MetricType type) {
        Series_.Clear();
        Y_ENSURE_EX(numId, yexception() << "NumId can not be zero");
        if (NumId_ != numId) {
            OnFinishShard();
            NumId_ = numId;
            Meta_ = MakeHolder<TMeta>(numId);
            Data_ = MakeHolder<TData>(numId);
        }

        LocalId_  = localId;
        Type_ = type;
    }

    void OnPoint(const yandex::solomon::stockpile::TPoint& point) {
        Y_ENSURE_EX(Data_, yexception() << "OnMetricBegin not called");
        if (!Data_->HasCommonTime) {
            Data_->Encoder->OnCommonTime(TInstant::MilliSeconds(point.timestampsmillis()));
            Data_->Encoder->OnStep(TDuration::Seconds(5));
            Data_->HasCommonTime = true;
        }

        Series_.Add(MakePoint(point, Type_));
    }

    size_t OnMetricEnd() {
        size_t points = Series_.Size();
        if (points == 0) {
            return points;
        }

        auto pointType = ToPointType(Type_);
        auto byteSize = Data_->WritePoints(pointType, Series_);
        Meta_->Encoder->OnMetric(pointType, LocalId_, points, byteSize);
        Series_.Clear();
        return points;
    }

    TSlogResult Finish() {
        OnFinishShard();

        TString encodeIndex;
        TStringOutput out(encodeIndex);
        NSolomon::NSlog::EncodeIndex(&out, Index_);
        out.Finish();
        Bytes_ = encodeIndex.size() + Result_.size();
        return {std::move(encodeIndex), std::move(Result_)};
    }

    size_t Points() const {
        return Points_;
    }

    size_t Bytes() const {
        return Bytes_;
    }

    size_t Metrics() const {
        return Metrics_;
    }

private:
    void OnFinishShard() {
        if (!Meta_ || Meta_->Metrics == 0) {
            Meta_.Reset();
            Data_.Reset();
            return;
        }

        Meta_->Finish();
        Data_->Finish();

        Index_.Add(NumId_, Meta_->Content.size(), Data_->Content.Size());
        Result_.append(Meta_->Content);
        Result_.append(Data_->Content);
        Points_ += Data_->Points;
        Metrics_ += Data_->Metrics;
        Bytes_ = Result_.Size();

        Meta_.Reset();
        Data_.Reset();
    }

private:
    TNumId NumId_{0};
    TLocalId LocalId_{0};
    yandex::solomon::model::MetricType Type_{yandex::solomon::model::MetricType::METRIC_TYPE_UNSPECIFIED};
    NSolomon::TAggrTimeSeries Series_;

    TString Result_{};
    size_t Points_{0};
    size_t Metrics_{0};
    size_t Bytes_{0};

    THolder<TMeta> Meta_;
    THolder<TData> Data_;

    NSolomon::NSlog::TSlogIndex Index_{0};
};

} // NHistDb::NStockpile
