#include "slog_builder.h"

#include <solomon/libs/cpp/slog/resolved_meta/iterator.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NHistDb::NStockpile;
using namespace yandex::solomon::stockpile;
using namespace yandex::solomon::model;
using namespace NSolomon::NSlog;
using namespace NSolomon::NTsModel;

Y_UNIT_TEST_SUITE(TSlogBuilderTest) {
    static TString SliceMeta(const TString& content, const TSlogIndex& index, size_t pos) {
        size_t offset = 0;
        for (size_t i = 0; i < pos; ++i) {
            offset += index.GetMetaSizeBytes(i);
            offset += index.GetDataSizeBytes(i);
        }
        return content.substr(offset, offset + index.GetMetaSizeBytes(pos));
    }

    static TString SliceData(const TString& content, const TSlogIndex& index, size_t pos) {
        size_t offset = index.GetMetaSizeBytes(pos);
        for (size_t i = 0; i < pos; ++i) {
            offset += index.GetMetaSizeBytes(i);
            offset += index.GetDataSizeBytes(i);
        }
        return content.substr(offset, offset + index.GetDataSizeBytes(pos));
    }

    Y_UNIT_TEST(OnePoint) {
        TInstant now = TInstant::Seconds(TInstant::Now().Seconds());
        TPoint proto;
        proto.set_timestampsmillis(now.MilliSeconds());
        proto.MutableSummaryDouble()->set_sum(1);
        proto.MutableSummaryDouble()->set_min(2);
        proto.MutableSummaryDouble()->set_max(3);
        proto.MutableSummaryDouble()->set_count(4);
        proto.MutableSummaryDouble()->set_last(5);

        TSlogResult result;
        {
            TSlogBuilder builder;
            builder.OnMetricBegin(42, 123, MetricType::DSUMMARY);
            builder.OnPoint(proto);
            builder.OnMetricEnd();
            result = builder.Finish();
        }

        TStringInput indexIn(result.Index);
        auto index = DecodeIndex(&indexIn);
        UNIT_ASSERT_EQUAL(1, index.Size());
        UNIT_ASSERT_EQUAL(42, index.GetNumId(0));

        {
            TString meta = SliceMeta(result.Content, index, 0);
            auto metaIt = NResolvedMeta::CreateResolvedMetaIterator(meta);

            UNIT_ASSERT_EQUAL(42, metaIt->NumId());
            UNIT_ASSERT_EQUAL(1, metaIt->TotalPointCount());
            UNIT_ASSERT_EQUAL(1, metaIt->TotalMetricCount());
            UNIT_ASSERT_EQUAL(true, metaIt->HasNext());
            auto record = metaIt->Next();
            UNIT_ASSERT_EQUAL(123, record.LocalId);
            UNIT_ASSERT_EQUAL(1, record.PointCounts);
            UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Type);
            UNIT_ASSERT_EQUAL(false, metaIt->HasNext());
        }

        {
            TString data = SliceData(result.Content, index, 0);
            TStringInput in(data);
            auto dataIt = CreateLogDataIterator(&in);

            UNIT_ASSERT_EQUAL(42, dataIt->NumId());
            UNIT_ASSERT_EQUAL(true, dataIt->HasNext());
            auto record = dataIt->Next();
            UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Kind);
            UNIT_ASSERT_EQUAL(1, record.TimeSeries.Size());
            UNIT_ASSERT_EQUAL(now, record.TimeSeries[0].GetTime());
            UNIT_ASSERT_EQUAL(1, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetSum());
            UNIT_ASSERT_EQUAL(2, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetMin());
            UNIT_ASSERT_EQUAL(3, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetMax());
            UNIT_ASSERT_EQUAL(4, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetCount());
            UNIT_ASSERT_EQUAL(5, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetLast());
            UNIT_ASSERT_EQUAL(false, dataIt->HasNext());
        }
    }

    Y_UNIT_TEST(ManyPoint) {
        TInstant now = TInstant::Seconds(TInstant::Now().Seconds());
        TSlogResult result;
        {
            TPoint proto;

            TSlogBuilder builder;
            builder.OnMetricBegin(43, 222, MetricType::DSUMMARY);

            proto.set_timestampsmillis(now.MilliSeconds());
            proto.MutableSummaryDouble()->set_sum(1);
            builder.OnPoint(proto);

            proto.set_timestampsmillis(now.MilliSeconds() + 5000);
            proto.MutableSummaryDouble()->set_sum(2);
            builder.OnPoint(proto);

            builder.OnMetricEnd();
            result = builder.Finish();
        }

        TStringInput indexIn(result.Index);
        auto index = DecodeIndex(&indexIn);
        UNIT_ASSERT_EQUAL(1, index.Size());
        UNIT_ASSERT_EQUAL(43, index.GetNumId(0));

        {
            TString meta = SliceMeta(result.Content, index, 0);
            auto metaIt = NResolvedMeta::CreateResolvedMetaIterator(meta);

            UNIT_ASSERT_EQUAL(43, metaIt->NumId());
            UNIT_ASSERT_EQUAL(2, metaIt->TotalPointCount());
            UNIT_ASSERT_EQUAL(1, metaIt->TotalMetricCount());
            UNIT_ASSERT_EQUAL(true, metaIt->HasNext());
            auto record = metaIt->Next();
            UNIT_ASSERT_EQUAL(222, record.LocalId);
            UNIT_ASSERT_EQUAL(2, record.PointCounts);
            UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Type);
            UNIT_ASSERT_EQUAL(false, metaIt->HasNext());
        }

        {
            TString data = SliceData(result.Content, index, 0);;
            TStringInput in(data);
            auto dataIt = CreateLogDataIterator(&in);

            UNIT_ASSERT_EQUAL(43, dataIt->NumId());
            UNIT_ASSERT_EQUAL(true, dataIt->HasNext());
            auto record = dataIt->Next();
            UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Kind);
            UNIT_ASSERT_EQUAL(2, record.TimeSeries.Size());
            UNIT_ASSERT_EQUAL(now, record.TimeSeries[0].GetTime());
            UNIT_ASSERT_EQUAL(1, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetSum());
            UNIT_ASSERT_EQUAL(now + TDuration::Seconds(5), record.TimeSeries[1].GetTime());
            UNIT_ASSERT_EQUAL(2, record.TimeSeries[1].GetValue().AsSummaryDouble()->GetSum());
            UNIT_ASSERT_EQUAL(false, dataIt->HasNext());
        }
    }

    Y_UNIT_TEST(ManyMetrics) {
        TInstant now = TInstant::Seconds(TInstant::Now().Seconds());
        TSlogResult result;
        {
            TPoint proto;

            TSlogBuilder builder;
            {
                builder.OnMetricBegin(43, 123, MetricType::DSUMMARY);
                proto.set_timestampsmillis(now.MilliSeconds());
                proto.MutableSummaryDouble()->set_sum(1);
                builder.OnPoint(proto);
                builder.OnMetricEnd();
            }
            {
                builder.OnMetricBegin(43, 321, MetricType::DSUMMARY);
                proto.set_timestampsmillis(now.MilliSeconds());
                proto.MutableSummaryDouble()->set_sum(2);
                builder.OnPoint(proto);
                builder.OnMetricEnd();
            }

            result = builder.Finish();
        }

        TStringInput indexIn(result.Index);
        auto index = DecodeIndex(&indexIn);
        UNIT_ASSERT_EQUAL(1, index.Size());
        UNIT_ASSERT_EQUAL(43, index.GetNumId(0));

        // meta
        {
            TString meta = SliceMeta(result.Content, index, 0);
            auto metaIt = NResolvedMeta::CreateResolvedMetaIterator(meta);

            UNIT_ASSERT_EQUAL(43, metaIt->NumId());
            UNIT_ASSERT_EQUAL(2, metaIt->TotalPointCount());
            UNIT_ASSERT_EQUAL(2, metaIt->TotalMetricCount());
            {
                UNIT_ASSERT_EQUAL(true, metaIt->HasNext());
                auto record = metaIt->Next();
                UNIT_ASSERT_EQUAL(123, record.LocalId);
                UNIT_ASSERT_EQUAL(1, record.PointCounts);
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Type);
            }
            {
                UNIT_ASSERT_EQUAL(true, metaIt->HasNext());
                auto record = metaIt->Next();
                UNIT_ASSERT_EQUAL(321, record.LocalId);
                UNIT_ASSERT_EQUAL(1, record.PointCounts);
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Type);
            }

            UNIT_ASSERT_EQUAL(false, metaIt->HasNext());
        }

        // data
        {
            TString data = SliceData(result.Content, index, 0);;
            TStringInput in(data);
            auto dataIt = CreateLogDataIterator(&in);

            UNIT_ASSERT_EQUAL(43, dataIt->NumId());
            {
                UNIT_ASSERT_EQUAL(true, dataIt->HasNext());
                auto record = dataIt->Next();
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Kind);
                UNIT_ASSERT_EQUAL(1, record.TimeSeries.Size());
                UNIT_ASSERT_EQUAL(now, record.TimeSeries[0].GetTime());
                UNIT_ASSERT_EQUAL(1, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetSum());
            }
            {
                UNIT_ASSERT_EQUAL(true, dataIt->HasNext());
                auto record = dataIt->Next();
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Kind);
                UNIT_ASSERT_EQUAL(1, record.TimeSeries.Size());
                UNIT_ASSERT_EQUAL(now, record.TimeSeries[0].GetTime());
                UNIT_ASSERT_EQUAL(2, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetSum());
            }
            UNIT_ASSERT_EQUAL(false, dataIt->HasNext());
        }
    }

    Y_UNIT_TEST(ManyShards) {
        TInstant now = TInstant::Seconds(TInstant::Now().Seconds());
        TSlogResult result;
        {
            TPoint proto;

            TSlogBuilder builder;
            {
                builder.OnMetricBegin(42, 123, MetricType::DSUMMARY);
                proto.set_timestampsmillis(now.MilliSeconds());
                proto.MutableSummaryDouble()->set_sum(1);
                builder.OnPoint(proto);
                builder.OnMetricEnd();
            }
            {
                builder.OnMetricBegin(52, 321, MetricType::DSUMMARY);
                proto.set_timestampsmillis(now.MilliSeconds());
                proto.MutableSummaryDouble()->set_sum(2);
                builder.OnPoint(proto);
                builder.OnMetricEnd();
            }

            result = builder.Finish();
        }

        TStringInput indexIn(result.Index);
        auto index = DecodeIndex(&indexIn);
        UNIT_ASSERT_EQUAL(2, index.Size());
        UNIT_ASSERT_EQUAL(42, index.GetNumId(0));
        UNIT_ASSERT_EQUAL(52, index.GetNumId(1));

        {
            // meta
            {
                TString meta = SliceMeta(result.Content, index, 0);
                auto metaIt = NResolvedMeta::CreateResolvedMetaIterator(meta);

                UNIT_ASSERT_EQUAL(42, metaIt->NumId());
                UNIT_ASSERT_EQUAL(1, metaIt->TotalPointCount());
                UNIT_ASSERT_EQUAL(1, metaIt->TotalMetricCount());

                UNIT_ASSERT_EQUAL(true, metaIt->HasNext());
                auto record = metaIt->Next();
                UNIT_ASSERT_EQUAL(123, record.LocalId);
                UNIT_ASSERT_EQUAL(1, record.PointCounts);
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Type);

                UNIT_ASSERT_EQUAL(false, metaIt->HasNext());
            }
            // data
            {
                TString data = SliceData(result.Content, index, 0);;
                TStringInput in(data);
                auto dataIt = CreateLogDataIterator(&in);

                UNIT_ASSERT_EQUAL(42, dataIt->NumId());

                UNIT_ASSERT_EQUAL(true, dataIt->HasNext());
                auto record = dataIt->Next();
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Kind);
                UNIT_ASSERT_EQUAL(1, record.TimeSeries.Size());
                UNIT_ASSERT_EQUAL(now, record.TimeSeries[0].GetTime());
                UNIT_ASSERT_EQUAL(1, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetSum());

                UNIT_ASSERT_EQUAL(false, dataIt->HasNext());
            }
        }
        {
            // meta 2
            {
                TString meta = SliceMeta(result.Content, index, 1);
                auto metaIt = NResolvedMeta::CreateResolvedMetaIterator(meta);

                UNIT_ASSERT_EQUAL(52, metaIt->NumId());
                UNIT_ASSERT_EQUAL(1, metaIt->TotalPointCount());
                UNIT_ASSERT_EQUAL(1, metaIt->TotalMetricCount());

                UNIT_ASSERT_EQUAL(true, metaIt->HasNext());
                auto record = metaIt->Next();
                UNIT_ASSERT_EQUAL(321, record.LocalId);
                UNIT_ASSERT_EQUAL(1, record.PointCounts);
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Type);

                UNIT_ASSERT_EQUAL(false, metaIt->HasNext());
            }
            // data 2
            {
                TString data = SliceData(result.Content, index, 1);
                TStringInput in(data);
                auto dataIt = CreateLogDataIterator(&in);

                UNIT_ASSERT_EQUAL(52, dataIt->NumId());

                UNIT_ASSERT_EQUAL(true, dataIt->HasNext());
                auto record = dataIt->Next();
                UNIT_ASSERT_EQUAL(EPointType::DSummary, record.Kind);
                UNIT_ASSERT_EQUAL(1, record.TimeSeries.Size());
                UNIT_ASSERT_EQUAL(now, record.TimeSeries[0].GetTime());
                UNIT_ASSERT_EQUAL(2, record.TimeSeries[0].GetValue().AsSummaryDouble()->GetSum());

                UNIT_ASSERT_EQUAL(false, dataIt->HasNext());
            }
        }
    }

}
