#include "state.h"

using namespace NHistDb::NStockpile;

TStockpileState::~TStockpileState() {
    Stop();
}

TMetabaseShardState TStockpileState::ResolveOneMetabaseShardForRead(TMetabaseShardKey key) {
    if (ReadingIsDisabled()) {
        ythrow TStockpileReadDisable() << "Metabase reading was disabled by fast config flag";
    }

    auto state = MetabaseShardProvider.FindOneShard(key);
    if (state.Defined()) {
        return state.GetRef();
    } else {
        ythrow TShardNotFoundError() << "There is no metabase shard " << key << " in cache";
    }
}

TMetabaseShardState TStockpileState::ResolveOneMetabaseShardForWrite(TMetabaseShardKey key) {
    if (WritingIsDisabled()) {
        ythrow TStockpileDumpDisable() << "Metabase writing was disabled by fast config flag";
    }

    return MetabaseShardProvider.ResolveOneShard(key);
}

TMaybe<TMetabaseShardState> TStockpileState::FindOneMetabaseShardForWrite(TMetabaseShardKey key) {
    TMaybe<TMetabaseShardState> result;
    if (!WritingIsDisabled()) {
        result = MetabaseShardProvider.FindOneShard(key);
    }
    return result;
}

TAtomicSharedPtr<TStockpileShard> TStockpileState::ResolveOneStockpileShardForRead(TStockpileShardId shardId) const {
    return StockpileShardProvider.ResolveOneShardForRead(shardId);
}

TAtomicSharedPtr<TStockpileShard> TStockpileState::ResolveOneStockpileShardForWrite(TStockpileShardId shardId) const {
    return StockpileShardProvider.ResolveOneShardForWrite(shardId);
}

EStockpileClusterType TStockpileState::GetClusterType() const {
    return Settings.ClusterInfo.ClusterType;
}

TShardIndex TStockpileState::GetShardsCount(TStringBuf itype) const {
    return Settings.MetabaseShardsConfig.GetShardsCount(itype);
}

void TStockpileState::UpdateShards() {
    MetabaseClusterProvider.TryToUpdate();
    StockpileClusterProvider.TryToUpdate();
    MetabaseShardProvider.TryToUpdateShards();
    StockpileShardProvider.TryToUpdateShards();
}

void TStockpileState::Start() {
    auto guard = Guard(ThreadMutex);
    Y_VERIFY(!Thread);

    Thread.Reset(new TThread(TThread::TParams(DoRun, this).SetName("lts_state_updater")));
    Thread->Start();
}

void TStockpileState::Stop() {
    auto guard = Guard(ThreadMutex);
    if (Thread) {
        AtomicSet(Done, 1);
        Event.Signal();
        Thread->Join();
        Thread.Reset();
    }
}

void* TStockpileState::DoRun(void* data) noexcept {
    TStockpileState* self = static_cast<TStockpileState*>(data);

    self->Log << TLOG_DEBUG << "Stockpile updater thread started";

    while (true) {
        self->Event.WaitT(TDuration::MilliSeconds(200));

        if (AtomicGet(self->Done)) {
            return nullptr;
        }

        try {
            self->OnTick();
        } catch (...) {
            self->Log << TLOG_ERR << "Stockpile updater thread failed with: " << CurrentExceptionMessage();
        }
    }
}

void TStockpileState::OnTick() {
    UpdateShards();
    MetabaseShardProvider.ProcessShardCreationQueue(TInstant::Now() + TDuration::Seconds(2));
};

TDataProxyMultiClusterState::TDataProxyMultiClusterState(EStockpileClusterType clusterType, TLog& log)
    : ClusterProvider(EStockpileDatabase::DataProxy, clusterType, log)
    , Thread()
    , Done(0) {
}

TDataProxyMultiClusterState::~TDataProxyMultiClusterState() {
    Stop();
}

void TDataProxyMultiClusterState::Start() {
    UpdateInfo();
    Thread.Reset(new TThread(DoRun, this));
    Thread->Start();
}

void TDataProxyMultiClusterState::Stop() {
    if (Thread) {
        AtomicSet(Done, 1);
        Thread->Join();
        Thread.Reset();
    }
}

void TDataProxyMultiClusterState::UpdateInfo() {
    ClusterProvider.TryToUpdate();
}

void* TDataProxyMultiClusterState::DoRun(void* data) noexcept {
    static_cast<TDataProxyMultiClusterState*>(data)->DoRun();
    return nullptr;
}

void TDataProxyMultiClusterState::DoRun() noexcept {
    const auto sleepDuration = TDuration::MilliSeconds(200);
    while (!AtomicGet(Done)) {
        Sleep(sleepDuration);
        UpdateInfo();
    }
}

TVector<TVector<TAtomicSharedPtr<TGrpcRemoteHost>>> TDataProxyMultiClusterState::GetAllHosts() const {
    return ClusterProvider.GetHostsGroupedByCluster();
}
