#include "stockpile_client.h"
#include "points.h"

#include <solomon/protos/stockpile/stockpile_requests.pb.h>
#include <solomon/protos/metabase/metric.pb.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

using namespace NHistDb::NStockpile;
using namespace yandex::solomon;
using yandex::solomon::model::MetricType;
using NZoom::NAccumulators::EAccumulatorType;

namespace {
    static constexpr TDuration STOCKPILE_TIMEOUT = TDuration::Minutes(1);

    template <typename TMethod>
    auto StockpileCallState(TMethod method, TAtomicSharedPtr<TGrpcRemoteHost> host, TGrpcRetryMode retryMode) -> TGrpcAsyncCallState<decltype(method)> {
        auto state(GrpcAsyncCallState(method, host, retryMode));
        state.SetDeadline(STOCKPILE_TIMEOUT);
        return state;
    }

    math::Aggregation ConvertAggregation(MetricType sensorType, EAccumulatorType aggregation) {
        using math::Aggregation;
        switch (sensorType) {
            case MetricType::LOG_HISTOGRAM:
            case MetricType::HIST:
            case MetricType::DSUMMARY: {
                return Aggregation::SUM;
            }
            default: {
                switch (aggregation) {
                    case EAccumulatorType::Last: {
                        return Aggregation::LAST;
                    }
                    case EAccumulatorType::Max: {
                        return Aggregation::MAX;
                    }
                    case EAccumulatorType::Min: {
                        return Aggregation::MIN;
                    }
                    case EAccumulatorType::Summ:
                    case EAccumulatorType::SummNone: {
                        return Aggregation::SUM;
                    }
                    case EAccumulatorType::Hgram: {
                        return Aggregation::SUM;
                    };
                    case EAccumulatorType::Average:
                    case EAccumulatorType::Avg: {
                        return Aggregation::SUM;
                    };
                    default: {
                        ythrow yexception() << "aggregation type " << aggregation << " does not supported in stockpile";
                    }
                }
            }
        };
    }
}

TPointHolder::TPointHolder(stockpile::TWriteRequest* request, TRecordSerializeState& recordSerializeState)
    : Request(request)
    , Finished(false)
    , RecordSerializeState(recordSerializeState)
{
}

TPointHolder::~TPointHolder() {
    if (!Finished) {
        Request->MutablePoints()->RemoveLast();
    }
}

void TPointHolder::Process(
        TInstant timestamp,
        NZoom::NValue::TValueRef value
) {
    TValueSerializer serializer(*Request->MutablePoints()->Add(), timestamp, RecordSerializeState);
    value.Update(serializer);
    if (serializer.Empty()) {
        if (serializer.Error()) {
            RecordSerializeState.IncErrorPointsCounter();
        }
        return;
    }
    Finished = true;
}

TWriteManyBuilder::TWriteManyBuilder(
        TStockpileShardId shardId,
        TAtomicSharedPtr<TGrpcRemoteHost> host,
        IStockpileClientStats& stats
)
    : CallState(StockpileCallState(
        &TStockpileService::Stub::PrepareAsyncWriteLog,
        host,
        EGrpcRetryModeFlags::ON_DEADLINE | EGrpcRetryModeFlags::ON_INTERNAL_ERROR))
    , Stats(stats)
{
    CallState.GetRequest().SetShardId(shardId);
    CallState.GetContext().set_compression_algorithm(GRPC_COMPRESS_NONE);
}

void TWriteManyBuilder::StartRecord(TRecordSerializeState&& state) {
    SlogBuilder_.OnMetricBegin(state.GetOwnerShardNumId(), state.GetSensorId().LocalId, state.GetType());
    CurrentRecordSerializeState = std::move(state);
}

void TWriteManyBuilder::AddPoint(TInstant timestamp, NZoom::NValue::TValueRef value) {
    ProtoPoint_.Clear();
    TValueSerializer serializer(ProtoPoint_, timestamp, CurrentRecordSerializeState.GetRef());
    value.Update(serializer);
    if (serializer.Empty()) {
        if (serializer.Error()) {
            CurrentRecordSerializeState->IncErrorPointsCounter();
        }
        return;
    }

    SlogBuilder_.OnPoint(ProtoPoint_);
}

size_t TWriteManyBuilder::Size() const {
    return SlogBuilder_.Points();
}

size_t TWriteManyBuilder::Metrics() const {
    return SlogBuilder_.Metrics();
}

size_t TWriteManyBuilder::Bytes() const {
    return SlogBuilder_.Bytes();
}

void TWriteManyBuilder::FinishRecord() {
    auto points = SlogBuilder_.OnMetricEnd();
    auto errorPoints = CurrentRecordSerializeState->GetErrorPointsCounter();
    if (errorPoints > 0) {
        TUnistat::Instance().PushSignalUnsafe(NMetrics::ERROR_POINTS, errorPoints);
    }

    if (points == 0) {
        if (errorPoints > 0) {
            Stats.OnRecordsError(1);
        } else {
            Stats.OnRecordsEmpty(1);
        }
    } else {
        auto ugramDiff = CurrentRecordSerializeState->GetDifferentUgramBoundsCounter();
        if (ugramDiff > 0) {
            TUnistat::Instance().PushSignalUnsafe(NMetrics::DIFFERENT_UGRAM_BOUNDS, ugramDiff);
        }
        auto croppedUgrams = CurrentRecordSerializeState->GetCroppedUgramCounter();
        if (croppedUgrams > 0) {
            TUnistat::Instance().PushSignalUnsafe(NMetrics::CROPPED_UGRAMS, croppedUgrams);
        }
    }
}

void TWriteManyBuilder::FinishBuild() {
    auto result = SlogBuilder_.Finish();
    CallState.GetRequest().set_index(std::move(result.Index));
    CallState.GetRequest().set_content(std::move(result.Content));
}

void TWriteManyBuilder::SetDeadline() {
    CallState.GetRequest().SetDeadline(ToDeadlineMillis(STOCKPILE_TIMEOUT));
}

TAtomicSharedPtr<TGrpcRemoteHost> TStockpileClient::GetHost() {
    return Host;
}

TServerStatusCallState TStockpileClient::ServerStatus() {
    auto state = StockpileCallState(&TStockpileService::Stub::PrepareAsyncServerStatus, Host, TGrpcRetryMode());
    state.GetRequest().SetDeadline(ToDeadlineMillis(STOCKPILE_TIMEOUT));
    state.GetContext().set_compression_algorithm(GRPC_COMPRESS_NONE);
    return state;
}

TWriteManyBuilder TStockpileClient::WriteManyBuilder(const TStockpileShardId shardId, IStockpileClientStats& stats) {
    return TWriteManyBuilder(shardId, Host, stats);
}

TReadUncompressedManyCallState TStockpileClient::ReadMany(TStockpileShardId shardId, const TStockpileReadOptions& options) {
    using math::Aggregation;

    auto state = StockpileCallState(&TStockpileService::Stub::PrepareAsyncReadUncompressedMany, Host, TGrpcRetryMode());
    auto& request = state.GetRequest();
    request.SetShardId(shardId);
    for (const auto& sensor : options.Sensors) {
        if (shardId != sensor.ShardId) {
            ythrow yexception() << "sensor not belong to this shard";
        }
        request.MutableLocalIds()->Add(sensor.LocalId);
    }
    request.SetFromMillis(options.Start.MilliSeconds());
    request.SetToMillis(options.End.MilliSeconds());
    request.SetDeadline(ToDeadlineMillis(STOCKPILE_TIMEOUT));
    if (options.Downsampling.Defined()) {
        auto* downsampling = request.add_operations()->mutable_downsampling();
        downsampling->set_aggregation(ConvertAggregation(options.Downsampling->SensorType, options.Downsampling->Aggregation));
        downsampling->set_grid_millis(options.Downsampling->Grid.MilliSeconds());
        downsampling->set_fill_option(math::OperationDownsampling_FillOption_NONE);
    }
    if (options.Combining.Defined()) {
        auto* combining = request.add_operations()->mutable_combine();
        combining->set_aggregation(ConvertAggregation(options.Combining->SensorType, options.Combining->Aggregation));
    }
    return state;
}

TDeleteDataCallState TStockpileClient::DeleteData(const TStockpileDeleteOptions& options) {
    auto state = StockpileCallState(&TStockpileService::Stub::PrepareAsyncDeleteMetricData, Host,
        EGrpcRetryModeFlags::ON_DEADLINE | EGrpcRetryModeFlags::ON_INTERNAL_ERROR);
    auto& request = state.GetRequest();
    request.SetFromMillis(options.Start.MilliSeconds());
    request.SetToMillis(options.End.MilliSeconds());
    request.mutable_metric_id()->set_shard_id(options.Sensor.ShardId);
    request.mutable_metric_id()->set_local_id(options.Sensor.LocalId);
    return state;
}


TStockpileStatusState::TStockpileStatusState(TGrpcCompletionQueue& queue, TAtomicSharedPtr<TGrpcRemoteHost> host)
    : Client(std::move(host))
    , CallState(Client.ServerStatus())
{
    queue.Execute(CallState, *this);
}

void TStockpileStatusState::Handle() {
    const auto& statusResponse(CallState.GetResponse());
    if (statusResponse.GetStatus() != stockpile::EStockpileStatusCode::OK) {
        ythrow yexception() << "Invalid Stockpile " << *Client.GetHost() << " ServerStatus response: \""
                            << statusResponse.GetStatusMessage() << "\" (" << statusResponse.GetStatus() << ")";
    }

    Shards.reserve(statusResponse.GetShardStatus().size());
    for (const auto& shardStatus : statusResponse.GetShardStatus()) {
        Shards.emplace_back(TShardStatus{
            .Key = shardStatus.GetShardId(),
            .ReadyWrite = shardStatus.GetReadyWrite(),
            .ReadyRead = shardStatus.GetReadyRead()
        });
    }
}

TVector<TStockpileStatusState::TShardStatus> TStockpileStatusState::GetResult() {
    CallState.Check();
    return Shards;
}

TAtomicSharedPtr<TGrpcRemoteHost> TStockpileStatusState::GetHost() {
    return Client.GetHost();
}
