#pragma once

#include "rpc.h"
#include "points.h"
#include "shard_cache.h"
#include "slog_builder.h"

#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/stockpile_client/common/base_types.h>

#include <solomon/protos/metabase/grpc_status.pb.h>
#include <solomon/protos/stockpile/stockpile_service.grpc.pb.h>

#include <library/cpp/logger/log.h>

#include <util/datetime/base.h>
#include <util/generic/vector.h>

namespace NHistDb::NStockpile {
    using TServerStatusCallState = TGrpcAsyncCallState<decltype(&TStockpileService::Stub::PrepareAsyncServerStatus)>;
    using TWriteManyCallState = TGrpcAsyncCallState<decltype(&TStockpileService::Stub::PrepareAsyncBulkShardCommand)>;
    using TWriteSlogCallState = TGrpcAsyncCallState<decltype(&TStockpileService::Stub::PrepareAsyncWriteLog)>;
    using TReadUncompressedManyCallState = TGrpcAsyncCallState<decltype(&TStockpileService::Stub::PrepareAsyncReadUncompressedMany)>;
    using TDeleteDataCallState = TGrpcAsyncCallState<decltype(&TStockpileService::Stub::PrepareAsyncDeleteMetricData)>;

    class TPointHolder {
    public:
        TPointHolder(yandex::solomon::stockpile::TWriteRequest* request, TRecordSerializeState& RecordSerializeState);

        ~TPointHolder();

        void Process(TInstant timestamp, NZoom::NValue::TValueRef value);

    private:
        yandex::solomon::stockpile::TWriteRequest* Request;
        bool Finished;
        TRecordSerializeState& RecordSerializeState;
    };

    class IStockpileClientStats {
    public:
        virtual void OnRecordsError(ui64 count) = 0;
        virtual void OnRecordsRejected(ui64 count) = 0;
        virtual void OnRecordsEmpty(ui64 count) = 0;
    };

    class TWriteManyBuilder {
    public:
        TWriteManyBuilder(
            TStockpileShardId shardId,
            TAtomicSharedPtr<TGrpcRemoteHost> host,
            IStockpileClientStats& stats
        );

        void StartRecord(TRecordSerializeState&& state);
        void AddPoint(TInstant timestamp, NZoom::NValue::TValueRef value);
        size_t Size() const;
        size_t Metrics() const;
        size_t Bytes() const;

        void FinishRecord();
        void FinishBuild();
        void SetDeadline();

        TWriteSlogCallState& GetCallState() {
            return CallState;
        }

    private:
        TWriteSlogCallState CallState;
        yandex::solomon::stockpile::TPoint ProtoPoint_{};
        TSlogBuilder SlogBuilder_{};
        IStockpileClientStats& Stats;
        TMaybe<TRecordSerializeState> CurrentRecordSerializeState;
    };

    struct TStockpileDownsamplingOptions {
        TDuration Grid;
        NZoom::NAccumulators::EAccumulatorType Aggregation;
        yandex::solomon::model::MetricType SensorType;
    };

    struct TStockpileCombiningOptions {
        NZoom::NAccumulators::EAccumulatorType Aggregation;
        yandex::solomon::model::MetricType SensorType;
    };

    struct TStockpileReadOptions {
        TVector<TSensorId> Sensors;
        TVector<NZoom::NAccumulators::EAccumulatorType> SensorTypes;
        TInstant Start;
        TInstant End;
        TMaybe<TStockpileDownsamplingOptions> Downsampling;
        TMaybe<TStockpileCombiningOptions> Combining;
        TMaybe<std::pair<TInstant, TInstant>> ResponsePointsWindow;
    };

    struct TStockpileDeleteOptions {
        TSensorId Sensor;
        TInstant Start;
        TInstant End;
    };

    class TStockpileClient {
    public:
        TStockpileClient(TAtomicSharedPtr<TGrpcRemoteHost> host)
            : Host(std::move(host))
        {
        }

        TAtomicSharedPtr<TGrpcRemoteHost> GetHost();

        TServerStatusCallState ServerStatus();
        TWriteManyBuilder WriteManyBuilder(TStockpileShardId shardId, IStockpileClientStats& stats);
        TReadUncompressedManyCallState ReadMany(TStockpileShardId shardId, const TStockpileReadOptions& options);
        TDeleteDataCallState DeleteData(const TStockpileDeleteOptions& options);

    private:
        TAtomicSharedPtr<TGrpcRemoteHost> Host;
    };

    class TStockpileStatusState final: public TGrpcState {
    public:
        using TShardStatus = TStockpileShardCache::TShardStatus;

        TStockpileStatusState(TGrpcCompletionQueue& queue, TAtomicSharedPtr<TGrpcRemoteHost> host);

        void Handle() override;

        TVector<TShardStatus> GetResult();
        TAtomicSharedPtr<TGrpcRemoteHost> GetHost();

        TStringBuf GetRequestName() const override {
            return NHistDb::NStockpile::NMetrics::STOCKPILE_GRPC_SERVER_STATUS;
        }

    private:
        TStockpileClient Client;

        TServerStatusCallState CallState;
        TVector<TShardStatus> Shards;
    };
}
