#include "stockpile_shard.h"
#include "points.h"

#include <solomon/protos/stockpile/stockpile_requests.pb.h>

using namespace NHistDb::NStockpile;
using namespace NZoom::NValue;
using NZoom::NAccumulators::EAccumulatorType;

namespace {
    void ThrowFailureException(yandex::solomon::stockpile::EStockpileStatusCode status, const TString& message,
        TGrpcRetryMode retryMode) {
        using yandex::solomon::stockpile::EStockpileStatusCode;
        switch (status) {
            case EStockpileStatusCode::SHARD_ABSENT_ON_HOST:
                ythrow TTopologyFailure() << message;
            case EStockpileStatusCode::SHARD_NOT_READY:
                ythrow TRetriableFailure() << message;
            case EStockpileStatusCode::DEADLINE_EXCEEDED: {
                if (retryMode.HasFlags(EGrpcRetryModeFlags::ON_DEADLINE)) {
                    ythrow TRetriableFailure() << message;
                } else {
                    ythrow yexception() << message;
                }
            }
            case EStockpileStatusCode::INTERNAL_ERROR:
            case EStockpileStatusCode::RESOURCE_EXHAUSTED: {
                if (retryMode.HasFlags(EGrpcRetryModeFlags::ON_INTERNAL_ERROR)) {
                    ythrow TRetriableFailure() << message;
                } else {
                    ythrow yexception() << message;
                }
            }
            default:
                ythrow yexception() << message;

        }
    }

    template<class TCallState>
    void CheckFailure(TCallState& callState, TStringBuf className, TStockpileShardId shardId,
        TAtomicSharedPtr<TGrpcRemoteHost> host, TGrpcRetryMode retryMode) {
        const auto& response(callState.GetResponse());
        if (response.GetStatus() != yandex::solomon::stockpile::EStockpileStatusCode::OK) {
            TString message = TStringBuilder() << "Invalid Stockpile " << *host << " " << className << " response: \""
                                               << response.GetStatusMessage() << "\" (" << response.GetStatus() << ")"
                                               << " ShardId=" << shardId;
            ThrowFailureException(response.GetStatus(), message, retryMode);
        }
    }
}

TStockpileWriteManyState::TStockpileWriteManyState(
        TAtomicSharedPtr<TStockpileShard> shard,
        TLog& log,
        IStockpileClientStats& stats
)
    : Client(shard->GetHost())
    , Log(log)
    , Shard(std::move(shard))
    , Builder(Client.WriteManyBuilder(Shard->GetShardId(), stats))
{
    Y_UNUSED(Log);
}

void TStockpileWriteManyState::Handle() {
    CheckFailure(Builder.GetCallState(), TStringBuf("WriteMany"), Shard->GetShardId(), Client.GetHost(),
        EGrpcRetryModeFlags::ON_DEADLINE | EGrpcRetryModeFlags::ON_INTERNAL_ERROR);
}

void TStockpileWriteManyState::StartRecord(TRecordSerializeState&& state) {
    if (state.GetSensorId().ShardId != Shard->GetShardId()) {
        ythrow yexception() << "point not belong to this shard";
    }
    Builder.StartRecord(std::move(state));
}

void TStockpileWriteManyState::AddPoint(TInstant timestamp, NZoom::NValue::TValueRef value) {
    Builder.AddPoint(timestamp, value);
}

void TStockpileWriteManyState::FinishRecord() {
    Builder.FinishRecord();
}

void TStockpileWriteManyState::FinishBuild() {
    Builder.FinishBuild();
}

bool TStockpileWriteManyState::Empty() {
    return Builder.Metrics() == 0;
}

size_t TStockpileWriteManyState::Size() const {
    return Builder.Size();
}

size_t TStockpileWriteManyState::Metrics() const {
    return Builder.Metrics();
}

size_t TStockpileWriteManyState::Bytes() const {
    return Builder.Bytes();
}

void TStockpileWriteManyState::ScheduleForExecute(TGrpcCompletionQueue& queue) {
    Builder.SetDeadline();
    StartTime = TInstant::Now();
    queue.Execute(Builder.GetCallState(), *this);
}

TStockpileReadManyState::TStockpileReadManyState(
    const TStockpileReadOptions& options,
    TAtomicSharedPtr<TStockpileShard> shard,
    TLog& log)
    : Client(shard->GetHost())
    , Log(log)
    , Shard(std::move(shard))
    , CallState(Client.ReadMany(Shard->GetShardId(), options))
    , ResponsePointsWindow(options.ResponsePointsWindow)
{
    Y_VERIFY(options.Sensors.size() == options.SensorTypes.size());

    SensorTypes.reserve(options.Sensors.size());
    for (const auto idx : xrange(options.Sensors.size())) {
        SensorTypes.emplace(options.Sensors[idx].LocalId, options.SensorTypes[idx]);
    }

    if (options.Combining) {
        CombiningAccumulator.ConstructInPlace(options.Combining->Aggregation);
    }

    Y_UNUSED(Log);
}

void TStockpileReadManyState::Handle() {
    CheckFailure(CallState, TStringBuf("ReadUncompressedMany"), Shard->GetShardId(), Client.GetHost(), TGrpcRetryMode());
}

grpc::ClientContext* TStockpileReadManyState::GetContext() {
    return &CallState.GetContext();
}

void TStockpileReadManyState::ScheduleForExecute(TGrpcCompletionQueue& queue) {
    StartTime = TInstant::Now();
    queue.Execute(CallState, *this);
}

void TStockpileReadManyState::VisitResult(ISensorValuesVisitor& visitor) {
    const auto& response(CallState.GetResponse());
    for (const auto& metric : response.metrics()) {
        visitor.OnSensorId(metric);
        if (metric.GetType() == yandex::solomon::model::MetricType::METRIC_TYPE_UNSPECIFIED) {
            continue;
        }

        NZoom::NAccumulators::EAccumulatorType accumulatorType;
        if (CombiningAccumulator.Defined()) {
            // it's combining result, take it's aggregation type
            accumulatorType = *CombiningAccumulator;
        } else {
            accumulatorType = SensorTypes.at(metric.GetLocalId());
        }
        TValueDeserializer decoder(metric.GetType(), accumulatorType);

        for (const auto& point : metric.GetUncompressed().GetPoints()) {
            auto time = TInstant::MilliSeconds(point.GetTimestampsMillis());
            if (!ResponsePointsWindow.Defined() || (ResponsePointsWindow->first <= time && time <= ResponsePointsWindow->second)) {
                visitor.OnValue(
                    time,
                    decoder.Decode(point)
                );
            }
        }
    }
}

TStockpileDeleteState::TStockpileDeleteState(const TStockpileDeleteOptions& options, TAtomicSharedPtr<TStockpileShard> shard)
    : Client(shard->GetHost())
    , Shard(std::move(shard))
    , CallState(Client.DeleteData(options))
{
}

void TStockpileDeleteState::Handle() {
    CheckFailure(CallState, TStringBuf("DeleteData"), Shard->GetShardId(), Client.GetHost(),
        EGrpcRetryModeFlags::ON_INTERNAL_ERROR | EGrpcRetryModeFlags::ON_DEADLINE);
}

grpc::ClientContext* TStockpileDeleteState::GetContext() {
    return &CallState.GetContext();
}

void TStockpileDeleteState::ScheduleForExecute(TGrpcCompletionQueue& queue) {
    queue.Execute(CallState, *this);
}

template <>
void Out<TStockpileShard>(IOutputStream& stream,
                          TTypeTraits<TStockpileShard>::TFuncParam shard) {
    stream << "TStockpileShard{.RemoteHost=" << *shard.GetHost() << "}";
}
