#pragma once

#include <infra/yasm/stockpile_client/common/base_types.h>
#include "metrics.h"
#include "stockpile_client.h"

#include <infra/yasm/common/points/value/types.h>

#include <library/cpp/logger/log.h>

namespace NHistDb::NStockpile {
    class TStockpileShard {
    public:
        TStockpileShard(TStockpileShardId shardId, TAtomicSharedPtr<TGrpcRemoteHost> stockpileHost, TLog& log)
            : ShardId(shardId)
            , GrpcRemoteHost(std::move(stockpileHost))
            , Log(log)
        {
            Y_UNUSED(Log);
        }

        TStockpileShardId GetShardId() const {
            return ShardId;
        }

        TAtomicSharedPtr<TGrpcRemoteHost> GetHost() const {
            return GrpcRemoteHost;
        }

    private:
        const TStockpileShardId ShardId;
        TAtomicSharedPtr<TGrpcRemoteHost> GrpcRemoteHost;
        TLog& Log;
    };

    class TStockpileWriteManyState final: public TGrpcState {
    public:
        TStockpileWriteManyState(TAtomicSharedPtr<TStockpileShard> shard, TLog& log, IStockpileClientStats& stats);

        void Handle() override;

        TStringBuf GetRequestName() const override {
            return NHistDb::NStockpile::NMetrics::STOCKPILE_GRPC_WRITE_MANY;
        }

        void StartRecord(TRecordSerializeState&& state);
        void AddPoint(TInstant timestamp, NZoom::NValue::TValueRef value);
        void FinishRecord();
        void FinishBuild();

        bool Empty();
        size_t Size() const;
        size_t Metrics() const;
        size_t Bytes() const;
        void ScheduleForExecute(TGrpcCompletionQueue& queue);

    private:
        TStockpileClient Client;
        TLog& Log;
        TAtomicSharedPtr<TStockpileShard> Shard;

        TWriteManyBuilder Builder;
    };

    class TRecordGuard : public TNonCopyable {
    public:
        TRecordGuard(TStockpileWriteManyState& writeState, TRecordSerializeState&& recordState)
            : WriteState(writeState)
        {
            WriteState.StartRecord(std::move(recordState));
        }

        ~TRecordGuard() {
            WriteState.FinishRecord();
        }

    private:
        TStockpileWriteManyState& WriteState;
    };

    class ISensorValuesVisitor {
    public:
        virtual ~ISensorValuesVisitor() = default;

        virtual void OnSensorId(const TSensorId& sensorId) = 0;
        virtual void OnValue(TInstant timestamp, NZoom::NValue::TValue value) = 0;
    };

    class TStockpileReadManyState final: public TGrpcState {
    public:
        TStockpileReadManyState(
            const TStockpileReadOptions& options,
            TAtomicSharedPtr<TStockpileShard> shard,
            TLog& log);

        void Handle() override;

        TStringBuf GetRequestName() const override {
            return NHistDb::NStockpile::NMetrics::STOCKPILE_GRPC_READ_MANY;
        }

        grpc::ClientContext* GetContext() override;

        void ScheduleForExecute(TGrpcCompletionQueue& queue);

        void VisitResult(ISensorValuesVisitor& visitor);

    private:
        TStockpileClient Client;
        TLog& Log;
        TAtomicSharedPtr<TStockpileShard> Shard;
        TReadUncompressedManyCallState CallState;
        THashMap<TLocalId, NZoom::NAccumulators::EAccumulatorType> SensorTypes;
        TMaybe<NZoom::NAccumulators::EAccumulatorType> CombiningAccumulator;
        TMaybe<std::pair<TInstant, TInstant>> ResponsePointsWindow;
    };

    class TStockpileDeleteState final: public TGrpcState {
    public:
        TStockpileDeleteState(
            const TStockpileDeleteOptions& options,
            TAtomicSharedPtr<TStockpileShard> shard);

        void Handle() override;

        TStringBuf GetRequestName() const override {
            return NHistDb::NStockpile::NMetrics::STOCKPILE_GRPC_DELETE_DATA;
        }

        grpc::ClientContext* GetContext() override;
        void ScheduleForExecute(TGrpcCompletionQueue& queue);
    private:
        TStockpileClient Client;
        TAtomicSharedPtr<TStockpileShard> Shard;
        TDeleteDataCallState CallState;
    };
}
