#include <infra/yasm/stockpile_client/common/base_types.h>
#include "stockpile_shard.h"
#include "stockpile_shard_provider.h"
#include "metrics.h"

#include <infra/monitoring/common/perf.h>

#include <util/generic/list.h>
#include <util/random/random.h>

using namespace NHistDb::NStockpile;

namespace {
    TDuration GetUpdateJitter() {
        return TDuration::FromValue(RandomNumber(STOCKPILE_UPDATE_JITTER.GetValue()));
    }
}

TStockpileShardProvider::TStockpileShardProvider(
        TClusterProvider& clusterProvider,
        ITopologyListener& listener,
        TLog& log)
    : ClusterProvider(clusterProvider)
    , Listener(listener)
    , Log(log)
{
}

TAtomicSharedPtr<TStockpileShard> TStockpileShardProvider::ResolveOneShardForRead(TStockpileShardId shardId) const {
    TLightReadGuard rg(ShardsLock);
    const auto& state(ResolveOneShard(shardId));
    if (!state.Status.ReadyRead) {
        ythrow TShardNotFoundError() << "Stockpile shard " << shardId << " not ready for reading";
    }
    return state.Shard;
}

TAtomicSharedPtr<TStockpileShard> TStockpileShardProvider::ResolveOneShardForWrite(TStockpileShardId shardId) const {
    TLightReadGuard rg(ShardsLock);
    const auto& state(ResolveOneShard(shardId));
    if (!state.Status.ReadyWrite) {
        ythrow TShardNotFoundError() << "Stockpile shard " << shardId << " not ready for writing";
    }
    return state.Shard;
}

const TStockpileShardProvider::TShardState& TStockpileShardProvider::ResolveOneShard(TStockpileShardId shardId) const {
    const auto it = CurrentShards.find(shardId);
    if (it.IsEnd()) {
        ythrow TShardNotFoundError() << "There is no stockpile shard " << shardId << " in cache";
    }
    return it->second;
}

void TStockpileShardProvider::TryToUpdateShards() {
    if (TInstant::Now() - LastUpdate >= STOCKPILE_UPDATE_INTERVAL) {
        UpdateShards();
        LastUpdate = TInstant::Now() - GetUpdateJitter();
    }
    TLightReadGuard rg(ShardsLock);
    TUnistat::Instance().PushSignalUnsafe(NMetrics::STOCKPILE_SHARD_CACHE_SIZE, CurrentShards.size());
}

void TStockpileShardProvider::UpdateShards() {
    TGrpcStateHandler handler(Log);
    TList<TStockpileStatusState> states;
    for (const auto& host : ClusterProvider.GetHosts()) {
        states.emplace_back(handler.GetQueue(), host);
    }
    handler.Wait();

    TStockpileShards newCurrentShards;
    TVector<TStockpileShardId> changedShards;
    for (auto& state : states) {
        try {
            const auto foundShards(state.GetResult());
            ShardCache.UpdateShardsForHost(*state.GetHost(), foundShards);
            Log << TLOG_DEBUG << foundShards.size() << " stockpile shards on " << *state.GetHost() << " found";
        } catch (...) {
            ShardCache.MarkHostAsFailed(*state.GetHost());
            Log << TLOG_ERR << "Stockpile host " << *state.GetHost() << " not working: " << CurrentExceptionMessage();
        }

        TLightReadGuard rg(ShardsLock);
        for (const auto& shardStatus : ShardCache.GetShardsForHost(*state.GetHost())) {
            const auto it(CurrentShards.find(shardStatus.Key));
            if (it == CurrentShards.end() || *it->second.Shard->GetHost() != *state.GetHost()) {
                // create new shard
                newCurrentShards.emplace(shardStatus.Key, TShardState{
                    .Shard = MakeAtomicShared<TStockpileShard>(shardStatus.Key, state.GetHost(), Log),
                    .Status = shardStatus
                });
                changedShards.emplace_back(shardStatus.Key);
                Log << TLOG_INFO << "New stockpile shard " << shardStatus << " on " << *state.GetHost();
            } else {
                // status may be updated, but shard with it's state is same
                newCurrentShards.emplace(shardStatus.Key, TShardState{
                    .Shard = it->second.Shard,
                    .Status = shardStatus
                });
                if (!(it->second.Status == shardStatus)) {
                    Log << TLOG_INFO << "Stockpile shard " << shardStatus << " on " << *state.GetHost() << " changed";
                }
            }
        }
    }
    ShardCache.Cleanup();

    Log << TLOG_INFO << "There are " << newCurrentShards.size() << " stockpile shards, " << changedShards.size() << " changed";

    {
        TLightWriteGuard rg(ShardsLock);
        CurrentShards.swap(newCurrentShards);
    }

    for (const auto& shardId : changedShards) {
        Listener.OnChangedStockpileShard(shardId);
    }
}
