#include "handlers.h"

#include <infra/monitoring/common/perf.h>
#include <infra/monitoring/common/msgpack.h>

#include "serialization.h"

using namespace NSubscriptions;
using namespace NMonitoring;
using namespace NZoom::NSubscription;

namespace {
    TVector<TSubscriptionWithRawKey> ParseSubscriptionsFromProtobuf(const TString& requestBody) {
        using google::protobuf::Arena;
        Arena arena;
        auto request = Arena::CreateMessage<NYasm::NInterfaces::NInternal::TSubscriptionListRequest>(&arena);
        Y_PROTOBUF_SUPPRESS_NODISCARD request->ParseFromArray(requestBody.data(), requestBody.size());

        NZoom::NProtobuf::THostNameDeserializer hostNameDeserializer(request->GetHostNameTable());
        NZoom::NProtobuf::TInternedRequestKeyDeserializer requestKeyDeserializer(request->GetRequestKeyTable());
        NZoom::NProtobuf::TSignalExpressionDeserializer signalNameDeserializer(request->GetSignalNameTable());
        NZoom::NProtobuf::TSubscriptionDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer,
            signalNameDeserializer);

        TVector<TSubscriptionWithRawKey> result(Reserve(request->SubscriptionsSize()));
        for (const auto& subscriptionProto: request->GetSubscriptions()) {
            auto [subscription, rawRequestKey] = deserializer.DeserializeWithFullInfo(subscriptionProto);
            result.emplace_back(TSubscriptionWithRawKey{
                .Subscription = subscription,
                .RawKey = rawRequestKey
            });
        }
        return result;
    }

    TVector<TSubscriptionWithRawKey> ParseSubscriptionsFromMsgPack(const TString& requestBody, const TStringBuf& path,
                                                                   TLog& logger) {
        auto oh = msgpack::unpack(requestBody.data(), requestBody.size());
        EnsureIs(oh.get(), msgpack::type::ARRAY);
        const auto& packedArray = oh.get().via.array;
        TVector<TSubscriptionWithRawKey> result(Reserve(packedArray.size));
        auto failedSubsCount = NSerialization::TraversePackedSubscriptionsArray(packedArray,
            [&result](const auto& host, const auto& reqKey, const auto& signal, bool allowLegacyTypes, auto&) {
                result.emplace_back(TSubscriptionWithRawKey{
                    .Subscription = TSubscription(host, reqKey, signal, allowLegacyTypes),
                    .RawKey = TString(reqKey)
                });
            });
        if (failedSubsCount) {
            logger << TLOG_WARNING << path << " failed to parse "
                   << failedSubsCount << " of " << packedArray.size << " subscriptions";
        }
        return result;
    }
}
namespace NSubscriptions {
    TVector<TSubscriptionWithRawKey> ParseSubscriptionsInRequest(const TString& requestBody, const THttpHeaders& requestHeaders,
                                                                 const TStringBuf& path, TLog& logger) {
        bool parseProtobuf = false;
        for (const auto& header : requestHeaders) {
            if (AsciiCompareIgnoreCase(header.Name(), "Content-Type") == 0) {
                parseProtobuf = header.Value().Contains(PROTOBUF_CONTENT_TYPE);
                break;
            }
        }

        if (parseProtobuf) {
            return ParseSubscriptionsFromProtobuf(requestBody);
        } else {
            return ParseSubscriptionsFromMsgPack(requestBody, path, logger);
        }
    }
}

void TPingHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_PING_TIME);
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("OK"));
}

size_t TAddManyHandler::HandleRequest(const NMonitoring::TServiceRequest::TRef& request, TInstant now,
                                      const TParsedHttpFull& meta) const {
    size_t created = 0;
    auto& input = request->GetInput();
    TVector<TSubscriptionWithRawKey> subscriptions = ParseSubscriptionsInRequest(input.ReadAll(), input.Headers(),
        meta.Path, Logger);
    for (auto& subscription: subscriptions) {
        if (Store.Add(subscription.Subscription, now)) {
            created++;
        }
    }
    return created;
}

void TAddManyHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_ADD_MANY_TIME);
    msgpack::sbuffer buffer;
    msgpack::packer<msgpack::sbuffer> packer(&buffer);
    packer.pack_map(1);
    PackString(packer, "created");
    packer.pack_uint64(HandleRequest(request, TInstant::Now(), meta));
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(TString(buffer.data(), buffer.size())));
}

TVector<THostName> TListManyHandler::DecodeRequest(TString body) const {
    TVector<THostName> result;
    auto oh = msgpack::unpack(body.data(), body.size());
    EnsureIs(oh.get(), msgpack::type::ARRAY);
    result.reserve(oh.get().via.array.size);
    for (const auto& host: TArrayIterator(oh.get().via.array)) {
        EnsureIs(host, msgpack::type::STR);
        result.emplace_back(host.as<TStringBuf>());
    }
    return result;
}

void TListManyHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_LIST_MANY_TIME);
    const auto hostNames(DecodeRequest(request->GetInput().ReadAll()));
    const auto reply(NSerialization::PackVersionedSubscriptions(Store.List(hostNames, TInstant::Now())));
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(reply));
}

void TListAllHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_LIST_ALL_TIME);
    const auto reply(NSerialization::PackVersionedSubscriptions(Store.ListAll(TInstant::Now())));
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(reply));
}

void TListAllNoGroupsHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_LIST_ALL_NO_GROUPS_TIME);
    const auto reply(NSerialization::PackVersionedSubscriptions(Store.ListAllNoGroups(TInstant::Now())));
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(reply));
}

void TFlushHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_FLUSH_TIME);
    Store.Flush();
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent("OK"));
}

void TGetRevisionHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull&) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_GET_REVISION_TIME);
    size_t revision = Store.GetRevision();
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(ToString(revision)));
}

TPushValuesHandler::TRequest TPushValuesHandler::DecodeRequest(TString body, const TParsedHttpFull& meta) const {
    TVector<TSubscriptionWithValue> values;
    TInstant timestamp;
    auto request = msgpack::unpack(body.data(), body.size());
    EnsureIs(request.get(), msgpack::type::MAP);

    for (const auto& requestField: TMapIterator(request.get().via.map)) {
        EnsureIs(requestField.key, msgpack::type::STR);
        TStringBuf fieldName = requestField.key.as<TStringBuf>();
        if (fieldName == ITERATION_TIMESTAMP_FIELD_NAME) {
            EnsureIs(requestField.val, msgpack::type::POSITIVE_INTEGER);
            timestamp = TInstant::Seconds(requestField.val.as<ui64>());
        } else if (fieldName == SUBSCRIPTION_LIST_FIELD_NAME) {
            EnsureIs(requestField.val, msgpack::type::ARRAY);
            const auto& packedArray = requestField.val.via.array;
            values.reserve(packedArray.size);
            auto failedSubsCount = NSerialization::TraversePackedSubscriptionsArray(packedArray,
                [&values](const auto& host, const auto& reqKey, const auto& signal, bool allowLegacyTypes, auto& ext) {
                    values.push_back(TSubscriptionWithValue{
                        .Subscription = TSubscription(host, reqKey, signal, allowLegacyTypes),
                        .Value = std::move(ext.SingleValue.GetRef())
                    });
            });

            if (failedSubsCount) {
                Logger << TLOG_WARNING << meta.Path << " failed to parse "
                       << failedSubsCount << " of " << packedArray.size << " subscriptions";
            }
        }
    }

    return TPushValuesHandler::TRequest{
        .Values = std::move(values),
        .Timestamp = timestamp
    };
}

void TPushValuesHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_PUSH_VALUES_TIME);

    TRequest requestParams = DecodeRequest(request->GetInput().ReadAll(), meta);

    auto pushed = Store.PushValues(std::move(requestParams.Values), requestParams.Timestamp);

    msgpack::sbuffer buffer;
    msgpack::packer<msgpack::sbuffer> packer(&buffer);
    packer.pack_map(1);
    PackString(packer, "pushed");
    packer.pack_uint64(pushed);
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(TString(buffer.data(), buffer.size())));
}

void TGetValuesHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_GET_VALUES_TIME);

    auto& input = request->GetInput();
    auto subscriptions = ParseSubscriptionsInRequest(input.ReadAll(), input.Headers(), meta.Path, Logger);
    auto now = TInstant::Now();
    for (auto& subscription: subscriptions) {
        Store.Add(subscription.Subscription, now);
    }

    bool respondInProtobuf = false;
    for (const auto& header : request->GetInput().Headers()) {
        if (AsciiCompareIgnoreCase(header.Name(), "Accept") == 0) {
            // respond in protobuf only if it was explicitly requested
            respondInProtobuf = header.Value().Contains(PROTOBUF_CONTENT_TYPE);
            break;
        }
    }

    if (respondInProtobuf) {
        TString responseBody = FillProtobufResponse(subscriptions);
        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(responseBody, PROTOBUF_CONTENT_TYPE));
    } else {
        TString responseBody = FillMsgPackResponse(subscriptions);
        request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(responseBody));
    }
}

TString TGetValuesHandler::FillMsgPackResponse(const TVector<TSubscriptionWithRawKey>& subscriptions) {
    msgpack::sbuffer buffer;
    msgpack::packer<msgpack::sbuffer> packer(&buffer);
    NSerialization::TMsgPackingVisitor visitor(packer, IncludeMessages);
    Store.VisitSubscriptionValues(subscriptions, visitor);
    return {buffer.data(), buffer.size()};
}

TString TGetValuesHandler::FillProtobufResponse(const TVector<TSubscriptionWithRawKey>& subscriptions) {
    using google::protobuf::Arena;
    Arena arena;
    auto response = Arena::CreateMessage<NYasm::NInterfaces::NInternal::TGetValuesResponse>(&arena);
    NZoom::NProtobuf::THostNameSerializer hostNameSerializer(response->MutableHostNameTable());
    NZoom::NProtobuf::TInternedRequestKeySerializer requesKeySerializer(response->MutableRequestKeyTable());
    NZoom::NProtobuf::TSignalExpressionSerializer signalNameSerializer(response->MutableSignalNameTable());
    NZoom::NProtobuf::TSubscriptionWithValueSeriesSerializer serializer(hostNameSerializer, requesKeySerializer,
        signalNameSerializer);

    bool includeRawRequestKeys = false;
    for (const auto& subscriptionWithRawKey: subscriptions) {
        if (subscriptionWithRawKey.Subscription.GetRequestKey().GetName() != subscriptionWithRawKey.RawKey) {
            includeRawRequestKeys = true;
            break;
        }
    }

    NSerialization::TProtoPackingVisitor visitor(serializer, response, includeRawRequestKeys);
    Store.VisitSubscriptionValues(subscriptions, visitor);
    return response->SerializeAsString();
}

void TSetValuesHandler::DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) {
    TMeasuredMethod perf(Logger, "", NMetrics::HANDLERS_SET_VALUES_TIME);

    TString body = request->GetInput().ReadAll();
    TDecodedRequest decodedRequest = DecodeRequest(body, meta);
    size_t setForSubsCount = Store.SetValues(std::move(decodedRequest.SubsWithValues));

    msgpack::sbuffer buffer;
    msgpack::packer<msgpack::sbuffer> packer(&buffer);
    packer.pack_map(1);
    PackString(packer, "set");
    packer.pack_uint64(setForSubsCount);
    request->Finish(THttpResponse(HttpCodes::HTTP_OK).SetContent(TString(buffer.data(), buffer.size())));
}

TVector<NZoom::NSubscription::TSubscriptionWithValueSeries> TSetValuesHandler::DecodeRequest(const msgpack::object_array& body,
                                                                                             const TParsedHttpFull& meta) const {
    TVector<TSubscriptionWithValueSeries> result;
    result.reserve(body.size);

    auto failedSubsCount = NSerialization::TraversePackedSubscriptionsArray(body,
        [&result, this](const auto& host, const auto& reqKey, const auto& signal, bool allowLegacyTypes, auto& extensions) {
            if (this->AcceptMessages && extensions.Messages.Defined()) {
                result.emplace_back(
                        TSubscription(host, reqKey, signal, allowLegacyTypes),
                        TValueSeries(extensions.ValueSeriesStartTimestamp.GetRef(),
                                     std::move(extensions.ValueSeries.GetRef())),
                        std::move(extensions.Messages.GetRef()));
            } else {
                result.emplace_back(
                        TSubscription(host, reqKey, signal, allowLegacyTypes),
                        TValueSeries(extensions.ValueSeriesStartTimestamp.GetRef(),
                                     std::move(extensions.ValueSeries.GetRef())));
            }
        });

    if (failedSubsCount) {
        Logger << TLOG_WARNING << meta.Path << " failed to parse "
               << failedSubsCount << " of " << body.size << " subscriptions";
    }
    return result;
}

TSetValuesHandler::TDecodedRequest TSetValuesHandler::DecodeRequest(TString body, const TParsedHttpFull& meta) const {
    auto request = msgpack::unpack(body.data(), body.size());
    EnsureIs(request.get(), msgpack::type::ARRAY);
    return TDecodedRequest(DecodeRequest(request.get().via.array, meta));
}
