#pragma once

#include "proto_handlers.h"
#include "metrics.h"

#include <infra/monitoring/common/application.h>
#include <infra/monitoring/common/msgpack.h>
#include <infra/yasm/zoom/components/subscription/store.h>

namespace NSubscriptions {

    constexpr TStringBuf PROTOBUF_CONTENT_TYPE = "application/x-protobuf";

    class TPingHandler : public NMonitoring::IServiceReplier {
    public:
        TPingHandler(TLog& logger)
            : Logger(logger) {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TLog& Logger;
    };

    class TAddManyHandler : public NMonitoring::IServiceReplier {
    public:
        TAddManyHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store)
            : Logger(logger)
            , Store(store)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        size_t HandleRequest(const NMonitoring::TServiceRequest::TRef& request, TInstant now, const TParsedHttpFull& meta) const;

        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
    };

    class TListManyHandler : public NMonitoring::IServiceReplier {
    public:
        TListManyHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store)
            : Logger(logger)
            , Store(store)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TVector<NZoom::NHost::THostName> DecodeRequest(TString body) const;

        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
    };

    class TListAllHandler : public NMonitoring::IServiceReplier {
    public:
        TListAllHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store)
            : Logger(logger)
            , Store(store)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
    };

    class TListAllNoGroupsHandler : public NMonitoring::IServiceReplier {
    public:
        TListAllNoGroupsHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store)
            : Logger(logger)
            , Store(store)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;
    private:
        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
    };

    class TFlushHandler : public NMonitoring::IServiceReplier {
    public:
        TFlushHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store)
            : Logger(logger)
            , Store(store)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
    };

    class TGetRevisionHandler : public NMonitoring::IServiceReplier {
    public:
        TGetRevisionHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store)
            : Logger(logger)
            , Store(store)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
    };

    class TPushValuesHandler : public NMonitoring::IServiceReplier {
    public:
        TPushValuesHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store)
            : Logger(logger)
            , Store(store)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        struct TRequest : public TMoveOnly {
            TVector<NZoom::NSubscription::TSubscriptionWithValue> Values;
            TInstant Timestamp;
        };

        TRequest DecodeRequest(TString body, const TParsedHttpFull& meta) const;

        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
    };

    class TGetValuesHandler : public NMonitoring::IServiceReplier {
    public:
        TGetValuesHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store, bool includeMessages)
            : Logger(logger)
            , Store(store)
            , IncludeMessages(includeMessages)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

        TString FillMsgPackResponse(const TVector<NZoom::NSubscription::TSubscriptionWithRawKey>& subscriptions);
        TString FillProtobufResponse(const TVector<NZoom::NSubscription::TSubscriptionWithRawKey>& subscriptions);

    private:
        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
        const bool IncludeMessages;
    };

    class TSetValuesHandler : public NMonitoring::IServiceReplier {
    public:
        TSetValuesHandler(TLog& logger, NZoom::NSubscription::TPurifyingStore& store, bool acceptMessages)
            : Logger(logger)
            , Store(store)
            , AcceptMessages(acceptMessages)
        {
        }

        void DoReply(const NMonitoring::TServiceRequest::TRef request, const TParsedHttpFull& meta) override;

    private:
        struct TDecodedRequest: public TMoveOnly {
            TDecodedRequest(TVector<NZoom::NSubscription::TSubscriptionWithValueSeries> subsWithValues)
                : SubsWithValues(std::move(subsWithValues)) {
            }
            TVector<NZoom::NSubscription::TSubscriptionWithValueSeries> SubsWithValues;
        };
        TDecodedRequest DecodeRequest(TString body, const TParsedHttpFull& meta) const;
        TVector<NZoom::NSubscription::TSubscriptionWithValueSeries> DecodeRequest(const msgpack::object_array& body,
                                                                                  const TParsedHttpFull& meta) const;

        TLog& Logger;
        NZoom::NSubscription::TPurifyingStore& Store;
        const bool AcceptMessages;
    };

    class TSubscriptionStoreHandlersCollection {
    public:
        TSubscriptionStoreHandlersCollection(TLog& logger, NYasm::NCommon::TFastConfig* fastConfig)
        {
            Store.SetFastConfig(fastConfig);

            // debug handler
            Handlers.emplace_back("/ping", MakeHolder<TPingHandler>(logger));

            // add_many handler creates or prolongs subscriptions
            Handlers.emplace_back("/add_many", MakeHolder<TAddManyHandler>(logger, Store));

            // list_many lists subscriptions for specified hosts
            Handlers.emplace_back("/list_many", MakeHolder<TListManyHandler>(logger, Store));

            // list_all lists all subscriptions
            Handlers.emplace_back("/list_all", MakeHolder<TListAllHandler>(logger, Store));

            // list_all_no_groups lists all subscriptions for hosts and containers. Group subscriptions are not included.
            Handlers.emplace_back("/list_all_no_groups", MakeHolder<TListAllNoGroupsHandler>(logger, Store));

            // Cleanup subscription store. Called from tests only.
            Handlers.emplace_back("/flush", MakeHolder<TFlushHandler>(logger, Store));

            // Get current revision of stores subscriptions
            Handlers.emplace_back("/get_revision", MakeHolder<TGetRevisionHandler>(logger, Store));

            // Push single values for multiple subscriptions.
            Handlers.emplace_back("/push_values", MakeHolder<TPushValuesHandler>(logger, Store));

            // Set full timeseries for multiple subscriptions.
            Handlers.emplace_back("/set_values", MakeHolder<TSetValuesHandler>(logger, Store, false));

            // Set full timeseries for multiple subscriptions. Attach messages to values (used in rtfront).
            Handlers.emplace_back("/set_values_with_msg", MakeHolder<TSetValuesHandler>(logger, Store, true));

            // Get values for specified subscriptions. Implicitly adds or prolongs subscriptions.
            Handlers.emplace_back("/get_values", MakeHolder<TGetValuesHandler>(logger, Store, false));

            // Get values and messages for specified subscriptions. Implicitly adds or prolongs subscriptions.
            Handlers.emplace_back("/get_values_with_msg", MakeHolder<TGetValuesHandler>(logger, Store, true));

            // Read data from subscription store in history api format. Exists for compatibility with tsdb and histdb-cacher formats.
            // Not used now.
            Handlers.emplace_back("/read_history", MakeHolder<TReadHistoryHandler>(Store, logger));
        }

        void Register(NMonitoring::TWebServer& server) {
            for (const auto& pair : Handlers) {
                server.Add(pair.first, *pair.second);
            }
        }

    private:
        NZoom::NSubscription::TPurifyingStore Store;
        TVector<std::pair<TString, THolder<NMonitoring::IServiceReplier>>> Handlers;
    };

    TVector<NZoom::NSubscription::TSubscriptionWithRawKey> ParseSubscriptionsInRequest(
            const TString& requestBody,
            const THttpHeaders& requestHeaders,
            const TStringBuf& path,
            TLog& logger);
}
