#include "handlers.h"

#include <library/cpp/testing/unittest/registar.h>

#include "serialization.h"

using namespace NMonitoring;
using namespace NZoom::NSubscription;
using namespace NSubscriptions;
using NZoom::NValue::TValue;

namespace {
    struct TRequestedSubscription {
        TString Host;
        TString Tags;
        TString Signal;
        bool AllowLegacyTypes = true;

        bool operator==(const TRequestedSubscription& other) const {
            return Host == other.Host &&
                   Tags == other.Tags &&
                   Signal == other.Signal &&
                   AllowLegacyTypes == other.AllowLegacyTypes;
        };
    };
    TString CreateSubscriptionsListRequest(const TVector<TRequestedSubscription>& subscriptions) {
        msgpack::sbuffer buffer;
        msgpack::packer<msgpack::sbuffer> packer(&buffer);
        packer.pack_array(subscriptions.size());
        for (const auto& subscription: subscriptions) {
            packer.pack_map(4);
            PackString(packer, HOST_FIELD_NAME);
            PackString(packer, subscription.Host);
            PackString(packer, TAGS_FIELD_NAME);
            PackString(packer, subscription.Tags);
            PackString(packer, SIGNAL_FIELD_NAME);
            PackString(packer, subscription.Signal);
            PackString(packer, ALLOW_LEGACY_TYPES_FIELD_NAME);
            if (subscription.AllowLegacyTypes) {
                packer.pack_true();
            } else {
                packer.pack_false();
            }
        }
        return {buffer.data(), buffer.size()};
    }

    TString CreateSubscriptionsListProtobufRequest(const TVector<TRequestedSubscription>& subscriptions) {
        NYasm::NInterfaces::NInternal::TSubscriptionListRequest request;
        NZoom::NProtobuf::THostNameSerializer hostNameSerializer(request.MutableHostNameTable());
        NZoom::NProtobuf::TSignalExpressionSerializer signalNameSerializer(request.MutableSignalNameTable());
        THashMap<TString, ui32> requestKeysToTableInd;

        for (const auto& subscription: subscriptions) {
            auto dest = request.MutableSubscriptions()->Add();
            hostNameSerializer.Intern(subscription.Host, dest->MutableHostName());
            signalNameSerializer.Intern(subscription.Signal, dest->MutableSignalExpression());

            auto [it, inserted] =
                requestKeysToTableInd.insert(std::make_pair(subscription.Tags, request.MutableRequestKeyTable()->NameSize()));
            if (inserted) {
                *request.MutableRequestKeyTable()->MutableName()->Add() = subscription.Tags;
            }
            dest->MutableRequestKey()->SetIndex(it->second);
        }

        return request.SerializeAsString();
    }
}

template <>
void Out<TRequestedSubscription>(IOutputStream& stream, TTypeTraits<TRequestedSubscription>::TFuncParam result) {
    stream << "TRequestedSubscription{Host=" << result.Host
           << ", Tags=" << result.Tags << ", Signal=" << result.Signal << ", AllowLegacyTypes=" << result.AllowLegacyTypes << "}";
}

Y_UNIT_TEST_SUITE(TestHandlers) {
    Y_UNIT_TEST(TestGetValuesForMultipleNames) {
        TPurifyingStore store;
        TLog logger;

        TGetValuesHandler handler(logger, store, false);

        TVector<TRequestedSubscription> requested = {
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=upper;ctype=prestable;prj=web-main-rkub;geo=sas",
                .Signal = "accesslog-5xx_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=upper;ctype=prestable;geo=sas;prj=web-main-rkub",
                .Signal = "accesslog-5xx_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "ctype=prestable;itype=upper;prj=web-main-rkub;geo=sas",
                .Signal = "accesslog-5xx_summ"
            }
        };
        TString request = CreateSubscriptionsListRequest(requested);

        TSubscription singleSubscription(
            TStringBuf("ASEARCH"),
            TStringBuf("ctype=prestable;itype=upper;prj=web-main-rkub;geo=sas"),
            TStringBuf("accesslog-5xx_summ")
        );

        auto startTime = TInstant::Now();
        UNIT_ASSERT(store.Add(singleSubscription, startTime));

        TVector<TSubscriptionWithValue> values;
        values.push_back(TSubscriptionWithValue{
            .Subscription = singleSubscription,
            .Value = TValue(1)
        });
        UNIT_ASSERT(store.PushValues(std::move(values), startTime));

        TString response = handler.FillMsgPackResponse(ParseSubscriptionsInRequest(request, THttpHeaders(),
            TStringBuf("/get_values"), logger));

        TVector<TRequestedSubscription> actuallyInResponse;

        auto oh = msgpack::unpack(response.data(), response.size());
        EnsureIs(oh.get(), msgpack::type::ARRAY);
        NSerialization::TraversePackedSubscriptionsArray(oh.get().via.array,
            [&](const auto& host, const auto& reqKey, const auto& signal, bool allowLegacyTypes, auto& extensions) {
                UNIT_ASSERT_VALUES_EQUAL(singleSubscription.GetRequestKey().GetName(), TString(reqKey));
                actuallyInResponse.emplace_back(TRequestedSubscription{
                    .Host = TString(host),
                    .Tags = TString(
                        (extensions.TagsAsRequested.Defined()) ? extensions.TagsAsRequested.GetRef() : reqKey),
                    .Signal = TString(signal),
                    .AllowLegacyTypes = allowLegacyTypes
                });
            });

        UNIT_ASSERT_VALUES_EQUAL(requested, actuallyInResponse);
    }

    Y_UNIT_TEST(TestGetValuesForMultipleNamesWithGaps) {
        TPurifyingStore store;
        TLog logger;

        TGetValuesHandler handler(logger, store, false);

        TVector<TRequestedSubscription> requested = {
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;prj=some;geo=sas",
                .Signal = "signal0_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;prj=other;geo=sas",
                .Signal = "signal1_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "ctype=prod;prj=some;geo=sas;itype=app",
                .Signal = "signal0_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;prj=other;geo=sas;ctype=prod",
                .Signal = "signal1_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;prj=some;geo=sas",
                .Signal = "signal0_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;geo=sas;prj=other",
                .Signal = "signal1_dhhh",
                .AllowLegacyTypes = false
            }
        };
        TString request = CreateSubscriptionsListRequest(requested);

        TSubscription singleSubscriptionWithValue(
            TStringBuf("ASEARCH"),
            TStringBuf("itype=app;ctype=prod;geo=sas;prj=other"),
            TStringBuf("signal1_summ")
        );
        TSubscription singleSubscriptionWithoutValue(
            TStringBuf("ASEARCH"),
            TStringBuf("itype=app;ctype=prod;geo=sas;prj=some"),
            TStringBuf("signal0_summ")
        );
        TSubscription singleSubscriptionConverting(
            TStringBuf("ASEARCH"),
            TStringBuf("itype=app;ctype=prod;geo=sas;prj=other"),
            TStringBuf("signal1_dhhh"),
            false
        );

        auto startTime = TInstant::Now();
        UNIT_ASSERT(store.Add(singleSubscriptionWithValue, startTime));
        UNIT_ASSERT(store.Add(singleSubscriptionWithoutValue, startTime));
        UNIT_ASSERT(store.Add(singleSubscriptionConverting, startTime));

        TVector<TSubscriptionWithValue> values;
        values.push_back(TSubscriptionWithValue{
            .Subscription = singleSubscriptionWithValue,
            .Value = TValue(1)
        });
        UNIT_ASSERT(store.PushValues(std::move(values), startTime));

        TVector<TSubscriptionWithValue> valuesSmallHgrams;
        valuesSmallHgrams.push_back(TSubscriptionWithValue{
            .Subscription = singleSubscriptionConverting,
            .Value = TValue(NZoom::NHgram::THgram::Small({1, 4, 5}, 2))
        });
        UNIT_ASSERT(store.PushValues(std::move(valuesSmallHgrams), startTime));

        TString response = handler.FillMsgPackResponse(ParseSubscriptionsInRequest(request, THttpHeaders(),
            TStringBuf("/get_values"), logger));

        TVector<TRequestedSubscription> expectedInResponse = {
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;prj=other;geo=sas",
                .Signal = "signal1_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;prj=other;geo=sas;ctype=prod",
                .Signal = "signal1_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;geo=sas;prj=other",
                .Signal = "signal1_dhhh",
                .AllowLegacyTypes = false
            }
        };
        TVector<TRequestedSubscription> actuallyInResponse;

        auto oh = msgpack::unpack(response.data(), response.size());
        EnsureIs(oh.get(), msgpack::type::ARRAY);
        NSerialization::TraversePackedSubscriptionsArray(oh.get().via.array,
            [&](const auto& host, const auto& reqKey, const auto& signal, bool allowLegacyTypes, auto& extensions) {
                UNIT_ASSERT_VALUES_EQUAL(singleSubscriptionWithValue.GetRequestKey().GetName(), TString(reqKey));
                actuallyInResponse.emplace_back(TRequestedSubscription{
                    .Host = TString(host),
                    .Tags = TString(
                        (extensions.TagsAsRequested.Defined()) ? extensions.TagsAsRequested.GetRef() : reqKey),
                    .Signal = TString(signal),
                    .AllowLegacyTypes = allowLegacyTypes
                });
            });

        UNIT_ASSERT_VALUES_EQUAL(expectedInResponse, actuallyInResponse);
    }

    Y_UNIT_TEST(TestGetValuesForMultipleNamesWithGapsAndProto) {
        TPurifyingStore store;
        TLog logger;

        TGetValuesHandler handler(logger, store, false);

        TVector<TRequestedSubscription> requested = {
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;prj=some;geo=sas",
                .Signal = "signal0_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;prj=other;geo=sas",
                .Signal = "signal1_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "ctype=prod;prj=some;geo=sas;itype=app",
                .Signal = "signal0_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;prj=other;geo=sas;ctype=prod",
                .Signal = "signal1_summ"
            },
            TRequestedSubscription{
                .Host = "ASEARCH",
                .Tags = "itype=app;ctype=prod;prj=some;geo=sas",
                .Signal = "signal0_summ"
            }
        };
        TString request = CreateSubscriptionsListProtobufRequest(requested);

        TSubscription singleSubscriptionWithValue(
            TStringBuf("ASEARCH"),
            TStringBuf("itype=app;ctype=prod;geo=sas;prj=other"),
            TStringBuf("signal1_summ")
        );
        TSubscription singleSubscriptionWithoutValue(
            TStringBuf("ASEARCH"),
            TStringBuf("itype=app;ctype=prod;geo=sas;prj=some"),
            TStringBuf("signal0_summ")
        );

        auto startTime = TInstant::Seconds(100500);
        UNIT_ASSERT(store.Add(singleSubscriptionWithValue, startTime));
        UNIT_ASSERT(store.Add(singleSubscriptionWithoutValue, startTime));

        TVector<TSubscriptionWithValue> values;
        values.push_back(TSubscriptionWithValue{
            .Subscription = singleSubscriptionWithValue,
            .Value = TValue(1)
        });
        UNIT_ASSERT(store.PushValues(std::move(values), startTime));

        THttpHeaders headers;
        headers.AddHeader("Content-Type", PROTOBUF_CONTENT_TYPE);

        TString response = handler.FillProtobufResponse(ParseSubscriptionsInRequest(request, headers, TStringBuf("/get_values"),
            logger));

        NYasm::NInterfaces::NInternal::TGetValuesResponse actualResponse;
        Y_PROTOBUF_SUPPRESS_NODISCARD actualResponse.ParseFromString(response);

        const auto& actualRawKeys = actualResponse.GetRawRequestKeys();
        UNIT_ASSERT_VALUES_EQUAL(TVector<TString>(actualRawKeys.begin(), actualRawKeys.end()),
            TVector<TString>({"itype=app;ctype=prod;prj=other;geo=sas", "itype=app;prj=other;geo=sas;ctype=prod"}));

        UNIT_ASSERT_VALUES_EQUAL(1, actualResponse.GetHostNameTable().name_size());
        UNIT_ASSERT_VALUES_EQUAL(1, actualResponse.GetRequestKeyTable().name_size());
        UNIT_ASSERT_VALUES_EQUAL(1, actualResponse.GetSignalNameTable().name_size());
        UNIT_ASSERT_VALUES_EQUAL(2, actualResponse.GetSubscriptions().size());

        NZoom::NProtobuf::THostNameDeserializer hostNameDeserializer(actualResponse.GetHostNameTable());
        NZoom::NProtobuf::TInternedRequestKeyDeserializer requestKeyDeserializer(actualResponse.GetRequestKeyTable());
        NZoom::NProtobuf::TSignalExpressionDeserializer signalNameDeserializer(actualResponse.GetSignalNameTable());
        NZoom::NProtobuf::TSubscriptionWithValueSeriesDeserializer deserializer(hostNameDeserializer, requestKeyDeserializer,
            signalNameDeserializer);

        for (const auto& subscriptionProto: actualResponse.GetSubscriptions()) {
            auto subscriptionWithValues = deserializer.Deserialize(subscriptionProto);
            UNIT_ASSERT_VALUES_EQUAL(singleSubscriptionWithValue, subscriptionWithValues.Subscription);
            UNIT_ASSERT_VALUES_EQUAL(startTime.Seconds(), subscriptionWithValues.ValueSeries.FirstValueTimestamp.Seconds());
            UNIT_ASSERT_VALUES_EQUAL(1, subscriptionWithValues.ValueSeries.Values.size());
            UNIT_ASSERT_VALUES_EQUAL(TValue(1), subscriptionWithValues.ValueSeries.Values.front());
        }
    }
}
