#include "proto_handlers.h"

#include <infra/yasm/zoom/components/serialization/history/history.h>

using namespace NSubscriptions;
using namespace NZoom::NProtobuf;
using namespace NZoom::NSubscription;

namespace {
    using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

    static constexpr size_t MAX_SUBSCRIPTION_COUNT = 10000;
    static constexpr TDuration READ_TIMEOUT = TDuration::Seconds(2);
    static constexpr TDuration RESOLUTION = TDuration::Seconds(TSubscriptionValues::TIMESTAMP_SECONDS_GRANULARITY);

    class THistoryResponseFactory final : public IValueSeriesVisitor {
    public:
        THistoryResponseFactory(const THistoryRequest& request)
            : Request(request)
            , Start(Request.Start)
            , StatusCode(EStatusCode::THistoryAggregatedSeries_EStatusCode_OK)
        {
        }

        void OnSize(size_t found) override {
            Y_VERIFY(found <= 1);
            if (!found) {
                StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND;
            }
        }

        void OnSubscriptionValues(const TSubscriptionWithRawKey&, TInstant startTs,
                                  const TSubscriptionValues::TValuesContainer& values,
                                  const TVector<TString>& messages) override {
            Start = Max(startTs, Request.Start);
            TInstant currentTimestamp(startTs);
            Values.reserve(values.size());
            for (const auto& value : values) {
                if (currentTimestamp >= Request.Start && currentTimestamp <= Request.End) {
                    Values.emplace_back(value.GetValue());
                }
                currentTimestamp += Request.Period;
            }
            if (!messages.empty()) {
                StatusCode = EStatusCode::THistoryAggregatedSeries_EStatusCode_PARTIAL;
            }
        }

        THistoryResponse CreaterResponse() {
            return THistoryResponse(Request, Start, std::move(Values), StatusCode);
        }

    private:
        const THistoryRequest& Request;
        TInstant Start;
        TVector<NZoom::NValue::TValue> Values;
        EStatusCode StatusCode;
    };
}

void TReadHistoryHandler::Handle(TProtoContext& context) {
    TInstant deadline(READ_TIMEOUT.ToDeadLine());
    if (context.Request.GetDeadlineTimestamp()) {
        deadline = Min(deadline, TInstant::Seconds(context.Request.GetDeadlineTimestamp()));
    }

    if (TInstant::Now() > deadline) {
        context.ReplyingContext.ResponceCode = HttpCodes::HTTP_SERVICE_UNAVAILABLE;
        return;
    }

    TInstant now(TInstant::Now());
    auto requests(THistoryRequest::FromProto(context.Request));
    THistoryResponseWriter responseWriter(context.Response);
    responseWriter.Reserve(requests.size());
    size_t subscriptionCount = 0;
    for (const auto& request : requests) {
        if (request.SignalName.IsOld()
                || request.Start > request.End
                || request.Period != RESOLUTION
                || request.End - request.Start >= RESOLUTION * TSubscriptionValues::SERIES_MAX_LENGTH) {
            THistoryResponse response(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
            responseWriter.Add(response);
            continue;
        } else if (TInstant::Now() > deadline) {
            THistoryResponse response(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_TIME_EXCEEDED);
            responseWriter.Add(response);
            continue;
        } else if (subscriptionCount > MAX_SUBSCRIPTION_COUNT) {
            THistoryResponse response(request, EStatusCode::THistoryAggregatedSeries_EStatusCode_LIMIT_EXCEEDED);
            responseWriter.Add(response);
            continue;
        }

        TSubscription subscription(request.HostName, request.RequestKey, request.SignalName.GetName());
        Store.Add(subscription, now);

        THistoryResponseFactory visitor(request);
        Store.VisitSubscriptionValues(TSubscriptionWithRawKey{subscription, ""}, visitor);
        responseWriter.Add(visitor.CreaterResponse());

        subscriptionCount++;
    }
}
