#include "proto_handlers.h"

#include <infra/monitoring/common/proto_replier_ut.h>
#include <infra/yasm/common/tests.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>
#include <infra/yasm/zoom/components/subscription/store.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NSubscriptions;
using namespace NZoom::NProtobuf;
using namespace NZoom::NSubscription;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using NYasm::NInterfaces::NInternal::THistoryReadAggregatedRequest;
using NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

namespace {
    static const THostName HOST_NAME(TStringBuf("sas1-1234.search.yandex.net"));
    static const TInternedRequestKey REQUEST_KEY(TStringBuf("itype=substore"));
    static const TSignalName SIGNAL_NAME(TStringBuf("signal_summ"));
    static constexpr TDuration PERIOD(TDuration::Seconds(5));

    class THelper {
    public:
        THelper()
            : Logger(NYasm::NCommon::NTest::CreateLog())
            , ReadHandler(Store, Logger)
        {
        }

        void RawRead(const THistoryReadAggregatedRequest& request, THistoryReadAggregatedResponse& response) {
            auto httpResponse = NMonitoring::THttpRequestBuilder()
                .FromProtoRequest(request)
                .Execute(ReadHandler);
            UNIT_ASSERT_VALUES_EQUAL(httpResponse.Code, 200);
            httpResponse.ToProto(response);
        }

        TVector<THistoryResponse> ReadMany(const TVector<THistoryRequest>& historyRequests) {
            THistoryReadAggregatedRequest request;
            THistoryReadAggregatedResponse response;

            {
                THistoryRequestWriter requestWriter(request);
                requestWriter.Reserve(historyRequests.size());
                for (const auto& historyRequest : historyRequests) {
                    requestWriter.Add(historyRequest);
                }
            }

            RawRead(request, response);
            return THistoryResponse::FromProto(response);
        }

        THistoryResponse ReadOne(const THistoryRequest& historyRequest) {
            auto responses(ReadMany(TVector<THistoryRequest>{historyRequest}));
            UNIT_ASSERT_VALUES_EQUAL(responses.size(), 1);
            return std::move(responses.back());
        }

        void AddSubscription(const TSubscription& subscription, TValueSeries valueSeries) {
            Store.Add(subscription, valueSeries.FirstValueTimestamp);
            TVector<TSubscriptionWithValueSeries> subs;
            subs.emplace_back(TSubscriptionWithValueSeries(subscription, std::move(valueSeries)));
            size_t updated = Store.SetValues(std::move(subs));
            UNIT_ASSERT_VALUES_EQUAL(updated, 1);
        }

    private:
        TLog Logger;
        TPurifyingStore Store;
        TReadHistoryHandler ReadHandler;
    };
}

Y_UNIT_TEST_SUITE(TestReadHistoryHandler) {
    Y_UNIT_TEST(TestSimpleReading) {
        THelper helper;

        size_t valueCount = 10;
        TInstant end = TInstant::Now();
        end -= TDuration::FromValue(end.GetValue() % PERIOD.GetValue());
        TInstant start(end - PERIOD * valueCount);

        TVector<TValue> values;
        for (const auto idx : xrange(valueCount)) {
            values.emplace_back(idx);
        }
        TSubscription subscription(HOST_NAME, REQUEST_KEY, SIGNAL_NAME.GetName());
        helper.AddSubscription(subscription, TValueSeries(start, std::move(values)));

        THistoryRequest request{
            .Start = start + PERIOD,
            .End = end - PERIOD,
            .Period = PERIOD,
            .HostName = HOST_NAME,
            .RequestKey = REQUEST_KEY,
            .SignalName = SIGNAL_NAME
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetStartTimestamp(), start + PERIOD);
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        TVector<TValue> expectedValues;
        for (const auto idx : xrange(valueCount - 1)) {
            expectedValues.emplace_back(idx + 1);
        }
        UNIT_ASSERT_VALUES_EQUAL(response.Series.GetValues(), expectedValues);
    }

    Y_UNIT_TEST(TestNotFound) {
        THelper helper;
        TInstant now = TInstant::Seconds(1000);
        THistoryRequest request{
            .Start = now,
            .End = now + PERIOD * (TSubscriptionValues::SERIES_MAX_LENGTH - 1),
            .Period = PERIOD,
            .HostName = HOST_NAME,
            .RequestKey = REQUEST_KEY,
            .SignalName = SIGNAL_NAME
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_NOT_FOUND);
    }

    Y_UNIT_TEST(TestInvalidPeriod) {
        THelper helper;
        THistoryRequest request{
            .Start = TInstant::Seconds(505),
            .End = TInstant::Seconds(505),
            .Period = TDuration::Seconds(1),
            .HostName = HOST_NAME,
            .RequestKey = REQUEST_KEY,
            .SignalName = SIGNAL_NAME
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
    }

    Y_UNIT_TEST(TestInvalidSignal) {
        THelper helper;
        THistoryRequest request{
            .Start = TInstant::Seconds(505),
            .End = TInstant::Seconds(505),
            .Period = PERIOD,
            .HostName = HOST_NAME,
            .RequestKey = REQUEST_KEY,
            .SignalName = TStringBuf("wrong")
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
    }

    Y_UNIT_TEST(TestInvalidRange) {
        THelper helper;
        TInstant now = TInstant::Seconds(1000);
        THistoryRequest request{
            .Start = now,
            .End = now + PERIOD * TSubscriptionValues::SERIES_MAX_LENGTH,
            .Period = PERIOD,
            .HostName = HOST_NAME,
            .RequestKey = REQUEST_KEY,
            .SignalName = SIGNAL_NAME
        };
        auto response(helper.ReadOne(request));
        UNIT_ASSERT_VALUES_EQUAL(response.StatusCode, EStatusCode::THistoryAggregatedSeries_EStatusCode_INTERNAL_ERROR);
    }
}
