#include "serialization.h"

#include <infra/monitoring/common/msgpack.h>
#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/zoom/components/serialization/common/msgpack_utils.h>

using namespace NMonitoring;
using namespace NZoom::NSubscription;
using namespace NZoom::NHgram;
using namespace NZoom::NPython;

namespace NSubscriptions::NSerialization {
    namespace {
        void PackThreeMainSubscriptionFields(msgpack::packer<msgpack::sbuffer>& packer, const TSubscription& subscription) {
            PackString(packer, HOST_FIELD_NAME);
            PackString(packer, subscription.GetHostName().GetName());
            PackString(packer, TAGS_FIELD_NAME);
            PackString(packer, subscription.GetRequestKey().GetName());
            PackString(packer, SIGNAL_FIELD_NAME);
            PackString(packer, subscription.GetSignalExpression());
            PackString(packer, ALLOW_LEGACY_TYPES_FIELD_NAME);
            if (subscription.GetAllowLegacyTypes()) {
                packer.pack_true();
            } else {
                packer.pack_false();
            }
        }

        template<typename ValueContainer>
        void PackSubscriptionValueSeries(msgpack::packer<msgpack::sbuffer>& packer,
                                         TInstant firstValueTimestamp,
                                         const ValueContainer& values) {
            PackString(packer, VALUE_SERIES_START_FIELD_NAME);
            packer.pack_int64(firstValueTimestamp.Seconds());
            PackString(packer, VALUE_SERIES_FIELD_NAME);
            packer.pack_array(values.size());
            TValueRefSerializer<msgpack::sbuffer, TUgramCompressor> valueSerializer(packer);
            for (const auto& value: values) {
                value.Update(valueSerializer);
            }
        }
    }

    void PackSubscription(msgpack::packer<msgpack::sbuffer>& packer, const TSubscriptionWithTime& subWithTime) {
        packer.pack_map(5);
        PackThreeMainSubscriptionFields(packer, subWithTime.Subscription);
        PackString(packer, TIMESTAMP_FIELD_NAME);
        packer.pack_int64(subWithTime.Timestamp.Seconds());
    }

    void PackSubscription(msgpack::packer<msgpack::sbuffer>& packer, const TSubscriptionWithValue& subWithValue) {
        packer.pack_map(5);
        PackThreeMainSubscriptionFields(packer, subWithValue.Subscription);
        PackString(packer, VALUE_FIELD_NAME);
        TValueRefSerializer<msgpack::sbuffer, TUgramCompressor> valueSerializer(packer);
        subWithValue.Value.Update(valueSerializer);
    }

    TString PackVersionedSubscriptions(const TVersionedSubscriptions& versionedSubs) {
        msgpack::sbuffer buffer;
        msgpack::packer<msgpack::sbuffer> packer(&buffer);
        packer.pack_map(2);
        PackString(packer, "revision");
        packer.pack_int64(versionedSubs.Revision);
        PackString(packer, SUBSCRIPTION_LIST_FIELD_NAME);
        packer.pack_array(versionedSubs.Subscriptions.size());
        for (const auto& subWithTime: versionedSubs.Subscriptions) {
            PackSubscription(packer, subWithTime);
        }
        return {buffer.data(), buffer.size()};
    }

    void TMsgPackingVisitor::OnSubscriptionValues(const TSubscriptionWithRawKey& subscriptionWithRawKey, TInstant startTimestamp,
                                                  const TSubscriptionValues::TValuesContainer& values,
                                                  const TVector<TString>& messages) {
        constexpr ui32 REQUIRED_FIELD_COUNT = 6;
        const auto& requestedWithKey = subscriptionWithRawKey.RawKey;
        const auto& subscription = subscriptionWithRawKey.Subscription;
        const bool requestedWithNormalizedKey = requestedWithKey == subscription.GetRequestKey().GetName();
        const bool addMessages = IncludeMessages && !messages.empty();

        auto fieldCount = REQUIRED_FIELD_COUNT;
        if (!requestedWithNormalizedKey) {
            ++fieldCount;
        }
        if (addMessages) {
            ++fieldCount;
        }

        Packer.pack_map(fieldCount);

        PackThreeMainSubscriptionFields(Packer, subscription);
        PackSubscriptionValueSeries(Packer, startTimestamp, values);

        if (!requestedWithNormalizedKey) {
            PackString(Packer, TAGS_REQUESTED_FIELD_NAME);
            PackString(Packer, requestedWithKey);
        }
        if (addMessages) {
            PackString(Packer, MESSAGES_FIELD_NAME);
            Packer.pack_array(messages.size());
            for (const auto& message: messages) {
                PackString(Packer, message);
            }
        }
    }

    void TMsgPackingVisitor::OnSize(size_t size) {
        Packer.pack_array(size);
    }

    void TProtoPackingVisitor::OnSubscriptionValues(const TSubscriptionWithRawKey& subscription, TInstant startTimestamp,
                                                    const TSubscriptionValues::TValuesContainer& values,
                                                    const TVector<TString>&) {
        Serializer.Intern(subscription.Subscription, startTimestamp, values, Response->MutableSubscriptions()->Add());
        if (IncludeRawRequestKeys) {
            Response->AddRawRequestKeys(subscription.RawKey);
        }
    }

    void TProtoPackingVisitor::OnSize(size_t size) {
        Response->MutableSubscriptions()->Reserve(size);
    }
}
