#pragma once

#include <util/generic/strbuf.h>
#include <infra/monitoring/common/msgpack.h>
#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/zoom/components/serialization/deserializers/msgpack.h>
#include <infra/yasm/zoom/components/serialization/zoom_to_protobuf/subscription.h>
#include <infra/yasm/interfaces/internal/subscription_store.pb.h>

#include <infra/yasm/zoom/components/subscription/store.h>

/**
 * Subscription-store request/response serialization utils.
 */
namespace NSubscriptions::NSerialization {
    TString PackVersionedSubscriptions(const NZoom::NSubscription::TVersionedSubscriptions& versionedSubs);

    void PackSubscription(msgpack::packer<msgpack::sbuffer>& packer, const NZoom::NSubscription::TSubscriptionWithTime& subWithTime);
    void PackSubscription(msgpack::packer<msgpack::sbuffer>& packer, const NZoom::NSubscription::TSubscriptionWithValue& subWithValue);

    struct TDeserializedSubscriptionExtensions: public TNonCopyable {
        TMaybe<TStringBuf> TagsAsRequested;
        TMaybe<NZoom::NValue::TValue> SingleValue;
        TMaybe<TVector<NZoom::NValue::TValue>> ValueSeries;
        TMaybe<TInstant> ValueSeriesStartTimestamp;
        TMaybe<TVector<TString>> Messages;
    };

    // Traverses the packed array of subscriptions and calls a visitor on every one of them.
    // Will try to read most of the subscriptions and return the number of subscriptions that were skipped.
    template <typename Visitor>
    size_t TraversePackedSubscriptionsArray(const msgpack::object_array& packedSubscriptions, Visitor visitor) {
        using namespace NMonitoring;
        using namespace NZoom::NSubscription;
        using namespace NZoom::NPython;

        size_t failedCount = 0;
        for (const auto& subscriptionDef: TArrayIterator(packedSubscriptions)) {
            try {
                TStringBuf hostName;
                TStringBuf requestKey;
                TStringBuf signalExpression;
                bool allowLegacyTypes = TSubscription::ALLOW_LEGACY_TYPES_DEFAULT;
                TDeserializedSubscriptionExtensions extensions;

                EnsureIs(subscriptionDef, msgpack::type::MAP);
                for (const auto& subscriptionField: TMapIterator(subscriptionDef.via.map)) {
                    EnsureIs(subscriptionField.key, msgpack::type::STR);
                    TStringBuf fieldName = subscriptionField.key.as<TStringBuf>();
                    if (fieldName == HOST_FIELD_NAME) {
                        EnsureIs(subscriptionField.val, msgpack::type::STR);
                        hostName = subscriptionField.val.as<TStringBuf>();
                    } else if (fieldName == TAGS_FIELD_NAME) {
                        EnsureIs(subscriptionField.val, msgpack::type::STR);
                        requestKey = subscriptionField.val.as<TStringBuf>();
                    } else if (fieldName == SIGNAL_FIELD_NAME) {
                        EnsureIs(subscriptionField.val, msgpack::type::STR);
                        signalExpression = subscriptionField.val.as<TStringBuf>();
                    } else if (fieldName == ALLOW_LEGACY_TYPES_FIELD_NAME) {
                        EnsureIs(subscriptionField.val, msgpack::type::BOOLEAN);
                        allowLegacyTypes = subscriptionField.val.as<bool>();
                    } else if (fieldName == VALUE_FIELD_NAME) {
                        extensions.SingleValue = TMsgPackValueHierarchy::DeserializeValue(subscriptionField.val, true);
                    } else if (fieldName == VALUE_SERIES_START_FIELD_NAME) {
                        EnsureIs(subscriptionField.val, msgpack::type::POSITIVE_INTEGER);
                        extensions.ValueSeriesStartTimestamp = TInstant::Seconds(subscriptionField.val.as<ui64>());
                    } else if (fieldName == VALUE_SERIES_FIELD_NAME) {
                        extensions.ValueSeries = TVector<NZoom::NValue::TValue>();

                        EnsureIs(subscriptionField.val, msgpack::type::ARRAY);
                        for (const auto& valueDef: TArrayIterator(subscriptionField.val.via.array)) {
                            extensions.ValueSeries->push_back(TMsgPackValueHierarchy::DeserializeValue(valueDef, true));
                        }
                    } else if (fieldName == TAGS_REQUESTED_FIELD_NAME) {
                        EnsureIs(subscriptionField.val, msgpack::type::STR);
                        extensions.TagsAsRequested = subscriptionField.val.as<TStringBuf>();
                    } else if (fieldName == MESSAGES_FIELD_NAME) {
                        extensions.Messages = TVector<TString>();
                        EnsureIs(subscriptionField.val, msgpack::type::ARRAY);
                        for (const auto& message: TArrayIterator(subscriptionField.val.via.array)) {
                            extensions.Messages->push_back(message.as<TString>());
                        }
                    }
                }
                if (hostName.empty() || requestKey.empty() || signalExpression.empty())
                    ythrow yexception() << "Incorrect subscription definition";

                if (extensions.ValueSeries.Defined() != extensions.ValueSeriesStartTimestamp.Defined())
                    ythrow yexception() << "Incorrect subscription value series definition";

                visitor(hostName, requestKey, signalExpression, allowLegacyTypes, extensions);
            }  catch (...) {
                ++failedCount;
            }
        }
        return failedCount;
    }

    struct TMsgPackingVisitor: public NZoom::NSubscription::IValueSeriesVisitor {
        msgpack::packer<msgpack::sbuffer>& Packer;
        bool IncludeMessages;

        explicit TMsgPackingVisitor(msgpack::packer<msgpack::sbuffer>& packer, bool includeMessages)
            : Packer(packer)
            , IncludeMessages(includeMessages) {
        };

        void OnSize(size_t size) final;
        void OnSubscriptionValues(
                const NZoom::NSubscription::TSubscriptionWithRawKey& subscription,
                TInstant startTimestamp,
                const NZoom::NSubscription::TSubscriptionValues::TValuesContainer& values,
                const TVector<TString>& messages) final;
    };

    struct TProtoPackingVisitor: public NZoom::NSubscription::IValueSeriesVisitor {
        NZoom::NProtobuf::TSubscriptionWithValueSeriesSerializer& Serializer;
        NYasm::NInterfaces::NInternal::TGetValuesResponse* Response;
        bool IncludeRawRequestKeys;

        explicit TProtoPackingVisitor(NZoom::NProtobuf::TSubscriptionWithValueSeriesSerializer& serializer,
                                      NYasm::NInterfaces::NInternal::TGetValuesResponse* response,
                                      bool includeRawRequestKeys)
            : Serializer(serializer)
            , Response(response)
            , IncludeRawRequestKeys(includeRawRequestKeys) {
        };

        void OnSize(size_t size) final;
        void OnSubscriptionValues(
                const NZoom::NSubscription::TSubscriptionWithRawKey& subscription,
                TInstant startTimestamp,
                const NZoom::NSubscription::TSubscriptionValues::TValuesContainer& values,
                const TVector<TString>& messages) final;
    };
}
