#include "serialization.h"

#include <infra/yasm/zoom/components/subscription/store.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NSubscriptions;
using namespace NSubscriptions::NSerialization;
using namespace NMonitoring;
using namespace NZoom::NSubscription;
using NZoom::NValue::TValue;

Y_UNIT_TEST_SUITE(TestSubscriptionSerialization) {
    Y_UNIT_TEST(TestSubscriptionsWithValue) {
        const TSubscription subscription1(
            TStringBuf("host1"),
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test1_summ")
        );
        const TSubscription subscription2(
            TStringBuf("host2"),
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test2_dhhh"),
            true
        );
        const TSubscription subscription3(
            TStringBuf("host2"),
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test2_dhhh"),
            false
        );
        TVector<TSubscriptionWithValue> subsVector;
        subsVector.push_back(TSubscriptionWithValue{
            .Subscription = subscription1,
            .Value = TValue(1),
        });
        subsVector.push_back(TSubscriptionWithValue{
            .Subscription = subscription2,
            .Value = TValue(NZoom::NHgram::THgram::Small({6, 9, 14, 88, 65}, 3)),
        });

        subsVector.push_back(TSubscriptionWithValue{
                .Subscription = subscription3,
                .Value = TValue(NZoom::NHgram::THgram::Small({6, 9, 14, 88, 65}, 3)),
        });

        msgpack::sbuffer buffer;
        msgpack::packer<msgpack::sbuffer> packer(&buffer);
        packer.pack_array(subsVector.size());
        for (const auto& subscription: subsVector) {
            PackSubscription(packer, subscription);
        }
        TString packed = {buffer.data(), buffer.size()};

        auto oh = msgpack::unpack(packed.data(), packed.size());
        EnsureIs(oh.get(), msgpack::type::ARRAY);

        TVector<TSubscriptionWithValue> unpackedSubsVector;
        TraversePackedSubscriptionsArray(oh.get().via.array,
            [&](const auto& host, const auto& reqKey, const auto& signal, bool allowLegacyTypes, auto& ext) {
                unpackedSubsVector.push_back(TSubscriptionWithValue{
                    .Subscription = TSubscription(host, reqKey, signal, allowLegacyTypes),
                    .Value = std::move(ext.SingleValue.GetRef())
                });
            });
        UNIT_ASSERT_EQUAL(subsVector, unpackedSubsVector);
    }

    Y_UNIT_TEST(TestSubscriptionsMsgPackingVisitor) {
        const THostName hostName1(TStringBuf("SAS.001"));
        const THostName hostName2(TStringBuf("SAS.002"));
        const TSubscription subscription1(
            hostName1,
            TInternedRequestKey(TStringBuf("itype=base")),
            TStringBuf("test1_summ")
        );
        const TSubscription subscription2(
            hostName2,
            TInternedRequestKey(TStringBuf("geo=sas;ctype=prod;itype=app")),
            TStringBuf("test2_summ"),
            true
        );
        const TSubscription subscription3(
            TStringBuf("SAS.003"),
            TInternedRequestKey(TStringBuf("itype=common")),
            TStringBuf("test3_summ"),
            false
        );

        TPurifyingStore store;

        auto startTime = TInstant::Zero();
        UNIT_ASSERT(store.Add(subscription1, startTime));
        UNIT_ASSERT(store.Add(subscription2, startTime));
        UNIT_ASSERT(store.Add(subscription3, startTime));

        TVector<TSubscriptionWithValue> values;
        values.push_back(TSubscriptionWithValue{
            .Subscription = subscription1,
            .Value = TValue(1)
        });
        values.push_back(TSubscriptionWithValue{
            .Subscription = subscription2,
            .Value = TValue(2)
        });
        UNIT_ASSERT(store.PushValues(std::move(values), startTime + TDuration::Seconds(10)));

        TVector<TSubscriptionWithRawKey> requestedSubscriptions = {
            TSubscriptionWithRawKey{
                .Subscription = subscription1,
                .RawKey = "key1"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription1,
                .RawKey = "key2"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription2,
                .RawKey = "key3"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription2,
                .RawKey = "key4"
            },
            TSubscriptionWithRawKey{
                .Subscription = subscription2,
                .RawKey = "key5"
            }
        };

        msgpack::sbuffer buffer;
        msgpack::packer<msgpack::sbuffer> packer(&buffer);
        NSerialization::TMsgPackingVisitor visitor(packer, true);
        store.VisitSubscriptionValues(requestedSubscriptions, visitor);
        TString packedByVisitor = {buffer.data(), buffer.size()};

        TVector<TValue> expectedValues10;
        TVector<TValue> expectedValues11;
        TVector<TValue> expectedValues20;
        TVector<TValue> expectedValues21;
        TVector<TValue> expectedValues22;
        expectedValues10.push_back(TValue(1));
        expectedValues11.push_back(TValue(1));
        expectedValues20.push_back(TValue(2));
        expectedValues21.push_back(TValue(2));
        expectedValues22.push_back(TValue(2));
        TVector<TSubscriptionWithValueSeries> expected;

        // expect to see subscription1 two times because we have given it two request keys
        expected.emplace_back(subscription1, TValueSeries(startTime + TDuration::Seconds(10), std::move(expectedValues10)));
        expected.emplace_back(subscription1, TValueSeries(startTime + TDuration::Seconds(10), std::move(expectedValues11)));
        expected.emplace_back(subscription2, TValueSeries(startTime + TDuration::Seconds(10), std::move(expectedValues20)));
        expected.emplace_back(subscription2, TValueSeries(startTime + TDuration::Seconds(10), std::move(expectedValues21)));
        expected.emplace_back(subscription2, TValueSeries(startTime + TDuration::Seconds(10), std::move(expectedValues22)));

        TVector<TSubscriptionWithValueSeries> actual;

        auto oh = msgpack::unpack(packedByVisitor.data(), packedByVisitor.size());
        EnsureIs(oh.get(), msgpack::type::ARRAY);
        TraversePackedSubscriptionsArray(oh.get().via.array,
            [&actual](const auto& host, const auto& reqKey, const auto& signal, bool allowLegacyTypes, auto& extensions) {
                actual.emplace_back(
                    TSubscription(host, reqKey, signal, allowLegacyTypes),
                    TValueSeries(extensions.ValueSeriesStartTimestamp.GetRef(), std::move(extensions.ValueSeries.GetRef())));
            });

        UNIT_ASSERT_EQUAL(expected, actual);
    }
}
