from util.generic.maybe cimport TMaybe
from util.generic.vector cimport TVector

from infra.yasm.unistat._unistat cimport (
    TUnistat,
    IHole, TAtomicSharedPtr, EAggregationType,
    CreatePriority, CreateValue
)
import six


def generate_intervals(base=1.5, min_log=-50, max_log=50):
    return [0.0] + [base ** x for x in xrange(min_log, max_log + 1)]


class AggregationType:
    Average = EAggregationType.Average
    HostHistogram = EAggregationType.HostHistogram
    Min = EAggregationType.Min
    Max = EAggregationType.Max
    Sum = EAggregationType.Sum
    SumOne = EAggregationType.SumOne
    LastValue = EAggregationType.LastValue


class SuffixType:
    Absolute = 'axxx'
    Sum = 'summ'
    Histogram = 'dhhh'


cdef class Metric:
    cdef TAtomicSharedPtr[IHole] hole

    def push(self, value):
        return self.hole.Get().PushSignal(value)

    def reset(self):
        self.hole.Get().ResetSignal()


cdef class Unistat:
    cdef TUnistat* unistat

    def __cinit__(self):
        self.unistat = new TUnistat()

    def __dealloc__(self):
        if self.unistat != NULL:
            del self.unistat

    @staticmethod
    def _format_tags(name, tags):
        prefix = ';'.join('{}={}'.format(k, v) for k, v in tags)
        return '{};{}'.format(prefix, name)

    def create_float(self, name, suffix=SuffixType.Sum, int priority=10, double start_value=0.0,
                     int aggregation_type=AggregationType.Sum, always_visible=False,
                     tags=()):
        if tags:
            name = self._format_tags(name, tags)
        metric = Metric()
        metric.hole = self.unistat.DrillFloatHole(
            six.b(name), six.b(suffix), CreatePriority(priority), CreateValue(start_value),
            <EAggregationType>aggregation_type, always_visible
        )
        return metric

    def create_histogram(self, name, suffix=SuffixType.Histogram, intervals=None, int priority=10, always_visible=False, tags=()):
        if intervals is None:
            intervals = generate_intervals()
        cdef TVector[double] wrapped_intervals = intervals
        if tags:
            name = self._format_tags(name, tags)
        metric = Metric()
        metric.hole = self.unistat.DrillHistogramHole(
            six.b(name), six.b(suffix), CreatePriority(priority), wrapped_intervals,
            EAggregationType.LastValue, always_visible
        )
        return metric

    def push(self, name, value, tags=()):
        if tags:
            name = self._format_tags(name, tags)

        return self.unistat.PushSignalUnsafe(six.b(name), value);

    def to_json(self, level=0, all_signals=False):
        return self.unistat.CreateJsonDump(level, all_signals)

    def to_info(self, level=0):
        return self.unistat.CreateInfoDump(level)

    def reset(self):
        self.unistat.Reset()

    def reset_signals(self):
        self.unistat.ResetSignals()

