# coding: utf-8

import json
import socket
import getpass
from six.moves.urllib.request import Request, urlopen
import logging


__all__ = ["USER_AGENT", "USER_AGENT_PREFIX",
           "HOSTNAME_HEADER", "USERNAME_HEADER",
           "Transport", "adjust_timestamp"]

USER_AGENT_PREFIX = "yasmapi/"
USER_AGENT = "arcadia yasmapi"
HOSTNAME_HEADER = "X-Golovan-Hostname"
USERNAME_HEADER = "X-Golovan-Username"
RT_HEADER = "X-Golovan-Rt-Request"

logger = logging.getLogger(__name__)


def adjust_timestamp(ts, period):
    return int(ts - (ts % period))


class Transport(object):
    """
    Класс реализует непосредственую коммуникацию с yasmfront.

    :param str golovan_host: Хост и опционально порт через `:` до yasmfront
    :param int connect_timeout: По истечении данного количества секунд
                                во время подключения к хосту, будет
                                поднято исключение socket.timeout
    """

    DEFAULT_GOLOVAN_HOST = "yasm.yandex-team.ru"
    DEFAULT_CONNECT_TIMEOUT = 25

    def __init__(self, golovan_host=None, connect_timeout=None):
        self.golovan_host = golovan_host if golovan_host else self.DEFAULT_GOLOVAN_HOST
        self.connect_timeout = connect_timeout if connect_timeout else self.DEFAULT_CONNECT_TIMEOUT

    def request(self, request, path, rt_hash=None):
        try:
            request = json.dumps(request)
        except Exception as exc:
            logger.error("Can't convert given request to JSON: %s", exc)
            request = json.dumps({})

        headers = {"Content-Type": "application/json",
                   "User-Agent": USER_AGENT,
                   HOSTNAME_HEADER: socket.gethostname()}

        try:
            headers[USERNAME_HEADER] = getpass.getuser()
        except Exception:
            pass

        if rt_hash is not None:
            headers[RT_HEADER] = str(rt_hash)

        req = Request("http://%s/%s/" % (self.golovan_host, path),
                              request.encode('utf-8'), headers)
        response = urlopen(req, timeout=self.connect_timeout)

        try:
            r = response.read()
            resp = json.loads(r.decode('utf-8'))
        except Exception as exc:
            logger.error("Can't parse server response: %s", exc)
            resp = {}

        return resp
