# coding: utf-8

from __future__ import division
import time
import hashlib
from collections import namedtuple, defaultdict
import six

from .core import Transport


class RtError(Exception):
    "Запрос для :class:`RtGolovanRequest` сформирован не правильно, или в процессе запроса к rtfront'у произошла ошибка"


class TsPoints(namedtuple("TsPoints", "ts values errors")):
    """Данные по одной realtime точке

    :param float ts: Timestamp точки
    :param dict values: значения по конкретным сигналам `{"host": {"tags": {"signal": val}}}`
    :param dict errors: известные ошибки, выявленные при вычислении сигналов в данной точке ({"signal": ["error"]})
    """


def _new_point_data(points, time_index):
    point = points[time_index]
    return point["value"], point.get("errors")


def _get_first_ts(response_list):
    return min(ts[0]["timestamp"]
               for response in response_list
               for ts in six.itervalues(response))


def _get_last_ts(response_list):
    return min(ts[-1]["timestamp"]
               for response in response_list
               for ts in six.itervalues(response))


def split_host_tags_signal(hts):
    key_tokens = hts.split(":")
    if len(key_tokens) != 3:
        raise RtError("Strange rtfront response: {}".format(hts))
    return key_tokens


class RtGolovanRequest(object):
    """
    Класс реализует интерфейс для запроса realtime данных из rtfront.
    Инстанс класса является итератором по объектам :class:`TsPoints`.
    Смотри :ref:`example_rt`.

    :param dict request_data: Запрашиваемыве сигналы должны быть в формате `{'host': {'tags': ['signal']}}`
    :param Transport transport: Транспорт для общения с yasmfront. Выбирая кастомный транспорт
                                можно изменить хост, таймаут. Потенциально можно изменить
                                протокол общения.
    """

    BATCH_SIZE = 200
    REQUEST_INTERVAL = 5
    RESOLUTION = 5
    GOLOVAN_PATH = "rt"

    def __init__(self, request_data, transport=None):
        if not isinstance(request_data, dict):
            raise RtError("'request_data' must have {'host': {'tag': ['signal']}} format")
        self.request_data = request_data
        self._processed_points = set()
        self.transport = transport or Transport()

    @staticmethod
    def _split_to_chunks(l, n):
        for idx in six.moves.range(0, len(l), n):
            yield l[idx:idx + n]

    @staticmethod
    def _compute_rt_hash(encoded_signals):
        hasher = hashlib.md5()
        for signal in encoded_signals:
            hasher.update(signal.encode('utf-8'))
        return int(hasher.hexdigest(), 16)

    @staticmethod
    def _convert_signals(request_data):
        return ["%s:%s:%s" % (host, tags, signal)
                for host, tags_signals in six.iteritems(request_data)
                for tags, signals in six.iteritems(tags_signals)
                for signal in signals]

    def _request_signals(self, encoded_signals):
        request = {"signals": encoded_signals}
        return self.transport.request(
            request, self.GOLOVAN_PATH,
            rt_hash=self._compute_rt_hash(encoded_signals)
        )

    def request(self, request=None):
        """Сделать  один запрос к tfront'у
        See https://wiki.yandex-team.ru/jandekspoisk/sepe/golovan/rtfront/http/#rt2 for details

        :param dict request: тело запроса (python структура)
        """
        rt_hash = None
        if request is None:
            encoded_signals = self._convert_signals(self.request_data)
            rt_hash = self._compute_rt_hash(encoded_signals)
            request = {"signals": encoded_signals}
        return self.transport.request(request, self.GOLOVAN_PATH, rt_hash=rt_hash)

    def _process_rt(self, response_list, first_ts, last_ts):
        """ format example:
        {
            "ASEARCH:base_self:loadlog-success": [
                {
                    "timestamp": 1360410500,
                    "value": 26344397,
                    "errors": [
                        "missing values for MAN1.0"
                    ]
                },
                ...
            ],
            ...
        }
        """
        for ts in six.moves.range(first_ts, last_ts + self.RESOLUTION, self.RESOLUTION):
            if ts in self._processed_points:
                continue
            result_values = defaultdict(lambda: defaultdict(dict))
            result_errors = defaultdict(lambda: defaultdict(dict))
            for response in response_list:
                for key, signal_info in six.iteritems(response):
                    delta = ts - signal_info[0]['timestamp']
                    if delta < 0:
                        continue
                    host, tags, signal = split_host_tags_signal(key)
                    time_index = delta // self.RESOLUTION
                    value, errors = _new_point_data(signal_info, time_index)
                    if value is not None:
                        result_values[host][tags][signal] = value
                    if errors:
                        result_errors[host][tags][signal] = errors
            if result_values:
                yield TsPoints(ts, result_values, result_errors)

    def points_from_request(self):
        """Do single request and extract all new points from it."""

        response_list = []
        encoded_signals = self._convert_signals(self.request_data)
        for signals_chunk in self._split_to_chunks(encoded_signals, self.BATCH_SIZE):
            response_with_status = self._request_signals(signals_chunk)
            if response_with_status["status"] != "ok":
                raise RtError(response_with_status)
            response_list.append(response_with_status["response"])

        first_ts = _get_first_ts(response_list)
        last_ts = _get_last_ts(response_list)

        for new_point in self._process_rt(response_list, first_ts, last_ts):
            self._processed_points.add(new_point.ts)
            yield new_point

        self._processed_points = {ts for ts in self._processed_points if ts >= first_ts}

    def __iter__(self, interval=None):
        """Generate realtime points in infinite loop
        Args:
            interval float: sleep time between request to rtfront. Default: 5
        """
        while True:
            for point in self.points_from_request():
                yield point
            time.sleep(interval or self.REQUEST_INTERVAL)
