#include "group.h"

#include <util/generic/hash_set.h>

using namespace NZoom::NAggregators;
using namespace NZoom::NContainers;
using namespace NZoom::NRecord;
using namespace NTags;

namespace {
    void CleanMap(THashMap<TInstanceKey, TGroupContainer>& map, const TInstant now) {
        for (auto it = map.begin(); it != map.end();) {
            auto prev = it;
            ++it;
            if (prev->second.Len()) {
                prev->second.Clean(now);
            }
            if (!prev->second.Len()) {
                map.erase(prev);
            }
        }
    }

    struct TTaggedRecordVisitor final: public ITagRecordCallback {
        TTaggedRecordVisitor(TTaggedMetricManager& metricManager, TGroupAggregator& aggregator)
            : MetricManager(metricManager)
            , Aggregator(aggregator)
        {
        }

        inline void OnTagRecord(TInstanceKey key, const TRecord& record) override {
            Aggregator.Mul(key, record, MetricManager);
        }

        TTaggedMetricManager& MetricManager;
        TGroupAggregator& Aggregator;
    };
}

void ITagGroupContainerCallback::SetObjectsCount(const size_t /* count */) {
}

TGroupAggregator::TGroupAggregator(const NZoom::NYasmConf::TYasmConf& conf)
    : Conf(conf)
{
}

TGroupContainer& TGroupAggregator::Find(NTags::TInstanceKey instanceKey) {
    TInstanceMap::insert_ctx ctx;
    auto newIt = Storage.find(instanceKey, ctx);
    if (newIt != Storage.end()) {
        return newIt->second;
    }
    auto it = Storage.emplace_direct(ctx, instanceKey, Conf.GetTypeConf(instanceKey.GetItype(), true));
    return it->second;
}

void TGroupAggregator::Mul(const NZoom::NRecord::TTaggedRecord& taggedRecord,
                           TTaggedMetricManager& metricManager) {
    TTaggedRecordVisitor visitor(metricManager, *this);
    taggedRecord.Process(visitor);
}

void TGroupAggregator::Mul(TInstanceKey instanceKey,
                           const NZoom::NRecord::TRecord& record,
                           TTaggedMetricManager& metricManager) {
    Find(instanceKey).Mul(record, metricManager.Create(instanceKey));
}

size_t TGroupAggregator::Len() const noexcept {
    return Storage.size();
}

void TGroupAggregator::Clean() {
    Clean(TInstant::Now());
}

void TGroupAggregator::Clean(const TInstant explicitNow) {
    CleanMap(Storage, explicitNow);
}

void TGroupAggregator::Process(ITagGroupContainerCallback& callback) const {
    callback.SetObjectsCount(Storage.size());
    for (const auto& p: Storage) {
        callback.OnTagContainer(p.first, p.second);
    }
}
