#include "history.h"

using namespace NZoom::NAggregators;
using namespace NZoom::NContainers;
using namespace NZoom::NProtobuf;
using namespace NZoom::NHost;
using namespace NZoom::NAccumulators;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

namespace {
    TMaybe<EAccumulatorType> GetAccumulatorType(const THistoryRequest& request, TAccumulatorMapping& accumulatorMapping) {
        return accumulatorMapping.GetAccumulatorType(
            request.RequestKey.GetRequestKey().GetItype(),
            request.SignalName
        );
    }
}

THistoryAggregator::TSeriesContainer::TSeriesContainer(const NZoom::NProtobuf::THistoryRequest& request,
                                                       NZoom::NContainers::TAccumulatorMapping& accumulatorMapping)
    : Container(GetAccumulatorType(request, accumulatorMapping), request.Start, request.End, request.Period, !request.HostName.IsGroup())
{
}

void THistoryAggregator::TSeriesContainer::Mul(const NZoom::NProtobuf::THistoryResponse& response) {
    TGuard<TAdaptiveLock> guard(Lock);
    Container.Mul(response.Series);
    StatusCodes[response.Request.HostName].emplace(response.StatusCode);
}

void THistoryAggregator::TSeriesContainer::Visit(IHistoryAggregatorVisitor& visitor) const {
    for (const auto& [hostName, statusCodes] : StatusCodes) {
        for (const auto& statusCode : statusCodes) {
            visitor.OnStatusCode(hostName, statusCode);
        }
    }
    visitor.OnContainer(Container);
}

THistoryAggregator::TOverwritingVisitor::TOverwritingVisitor(THistoryAggregator* aggregator, const IOverwritePolicy& policy)
    : Aggregator(aggregator)
    , Policy(policy)
{
}

void THistoryAggregator::TOverwritingVisitor::OnRequest(const THistoryRequest& request) {
    CurrentRequest = &request;

    auto seriesContainer(Aggregator->Containers.find(*CurrentRequest));
    if (seriesContainer != Aggregator->Containers.end()) {
        CurrentContainer = &seriesContainer->second;
    }
}

void THistoryAggregator::TOverwritingVisitor::OnStatusCode(THostName hostName, EStatusCode statusCode) {
    if (CurrentContainer) {
        CurrentContainer->StatusCodes[hostName].emplace(statusCode);
    }
}

void THistoryAggregator::TOverwritingVisitor::OnContainer(const TTimestampedSeriesContainer& container) {
    if (CurrentContainer) {
        CurrentContainer->Container.Overwrite(container, Policy);
    }
}

void THistoryAggregator::TOverwritingVisitor::OnFinish() {
    CurrentRequest = nullptr;
    CurrentContainer = nullptr;
}

void THistoryAggregator::AddRequest(const THistoryRequest& request) {
    Containers.emplace(request, TSeriesContainer(request, AccumulatorMapping));
}

void THistoryAggregator::AddMetagroup(NZoom::NHost::THostName metagroup, const TVector<NZoom::NHost::THostName>& groups) {
    for (const auto& group : groups) {
        GroupsToMetagroups[group].emplace(metagroup);
    }
}

void THistoryAggregator::Mul(const THistoryResponse& response) {
    THashSet<THistoryRequest> uniqueRequestsToMul; // mul every request's container only once
    uniqueRequestsToMul.insert(response.Request);
    auto metagroups = GroupsToMetagroups.find(response.Request.HostName);
    if (metagroups != GroupsToMetagroups.end()) {
        for (const auto& metagroup : metagroups->second) {
            THistoryRequest request(response.Request);
            request.HostName = metagroup;
            uniqueRequestsToMul.insert(request);
        }
    }

    for (const auto& request: uniqueRequestsToMul) {
        auto containerIt = Containers.find(request);
        if (containerIt != Containers.end()) {
            containerIt->second.Mul(response);
        }
    }
}

void THistoryAggregator::Overwrite(const THistoryAggregator& otherAggregator, const NZoom::NContainers::IOverwritePolicy& policy) {
    TOverwritingVisitor visitor(this, policy);
    otherAggregator.Visit(visitor);
}

void THistoryAggregator::Overwrite(const THistoryAggregator& otherAggregator) {
    TOverwriteAnythingPolicy policy;
    Overwrite(otherAggregator, policy);
}

void THistoryAggregator::OverwriteSince(const THistoryAggregator& otherAggregator, TInstant since, TInstant until) {
    TOverwriteWithSkipPolicy policy(since, until);
    Overwrite(otherAggregator, policy);
}

void THistoryAggregator::OverwriteContinuous(const THistoryAggregator& otherAggregator) {
    TOverwriteContinuousPolicy policy;
    Overwrite(otherAggregator, policy);
}

void THistoryAggregator::Visit(IHistoryAggregatorVisitor& visitor) const {
    for (const auto& [request, state] : Containers) {
        visitor.OnRequest(request);
        state.Visit(visitor);
        visitor.OnFinish();
    }
}
