#include "history.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NAggregators;
using namespace NZoom::NYasmConf;
using namespace NZoom::NProtobuf;
using namespace NZoom::NContainers;
using namespace NZoom::NValue;
using namespace NZoom::NRecord;
using namespace NZoom::NHost;
using EStatusCode = NYasm::NInterfaces::NInternal::THistoryAggregatedSeries::EStatusCode;

namespace {
    TYasmConf GetYasmConf() {
        return TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {}, \"patterns\": {}, \"periods\": {}}}}"));
    }

    TVector<TValue> CreateValues(double value, size_t count) {
        TVector<TValue> result(Reserve(count));
        for (const auto idx : xrange(count)) {
            result.emplace_back(value);
            Y_UNUSED(idx);
        }
        return result;
    }

    class TMockedContainerVisitor final : public ITimestampedSeriesContainerVisitor {
    public:
        void OnStartTime(TInstant startTime) override {
            StartTime = startTime;
        }

        void OnValueCount(size_t count) override {
            Values.reserve(count);
        }

        void OnValue(TValueRef value) override {
            Values.emplace_back(value);
        }

        TTimestampedNamedSeries Create(const THistoryRequest& request) {
            return TTimestampedNamedSeries(request.SignalName, std::move(Values), StartTime);
        }

        TInstant StartTime;
        TVector<TValue> Values;
    };

    class TMockedHistoryVisitor final : public IHistoryAggregatorVisitor {
    public:
        void OnRequest(const THistoryRequest& request) override {
            UNIT_ASSERT(!CurrentRequest);
            CurrentRequest = &request;
        }

        void OnStatusCode(THostName hostName, EStatusCode statusCode) override {
            UNIT_ASSERT(CurrentRequest);
            StatusCodes[*CurrentRequest][hostName] = statusCode;
        }

        void OnContainer(const TTimestampedSeriesContainer& container) override {
            UNIT_ASSERT(CurrentRequest);
            TMockedContainerVisitor visitor;
            container.Visit(visitor);
            Series.emplace(*CurrentRequest, visitor.Create(*CurrentRequest));
        }

        void OnFinish() override {
            CurrentRequest = nullptr;
            CurrentStatus = EStatusCode::THistoryAggregatedSeries_EStatusCode_UNKNOWN;
        }

        THashMap<THistoryRequest, TTimestampedNamedSeries> Series;
        THashMap<THistoryRequest, THashMap<THostName, EStatusCode>> StatusCodes;

        const THistoryRequest* CurrentRequest = nullptr;
        EStatusCode CurrentStatus = EStatusCode::THistoryAggregatedSeries_EStatusCode_UNKNOWN;
    };
}

Y_UNIT_TEST_SUITE(THistoryAggregatorTest) {
    Y_UNIT_TEST(Merge) {
        THistoryRequest hostRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        THistoryRequest firstGroupRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("SAS.000"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        THistoryRequest secondGroupRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("SAS.001"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        THistoryRequest metagroupRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("ASEARCH"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        THistoryRequest notFoundRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-not-exists"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };

        TYasmConf conf(GetYasmConf());
        THistoryAggregator aggregator(conf);
        aggregator.AddRequest(hostRequest);
        aggregator.AddRequest(firstGroupRequest);
        aggregator.AddRequest(metagroupRequest);
        aggregator.AddRequest(notFoundRequest);
        aggregator.AddMetagroup(TStringBuf("ASEARCH"), {TStringBuf("SAS.000")});
        aggregator.AddMetagroup(TStringBuf("ASEARCH"), {TStringBuf("SAS.001")});

        THistoryResponse hostResponse(
            hostRequest, TInstant::Seconds(100), CreateValues(1, 21), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        aggregator.Mul(hostResponse);

        THistoryResponse firstGroupResponse(
            firstGroupRequest, TInstant::Seconds(100), CreateValues(2, 21), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        aggregator.Mul(firstGroupResponse);

        THistoryResponse secondGroupResponse(
            secondGroupRequest, TInstant::Seconds(100), CreateValues(3, 21), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        aggregator.Mul(secondGroupResponse);

        TMockedHistoryVisitor visitor;
        aggregator.Visit(visitor);

        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(hostRequest).GetStartTimestamp(), hostRequest.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(hostRequest).GetValues(), CreateValues(1, 21));
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(hostRequest).at(TStringBuf("sas1-1234")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(firstGroupRequest).GetStartTimestamp(), firstGroupRequest.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(firstGroupRequest).GetValues(), CreateValues(2, 21));
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(firstGroupRequest).at(TStringBuf("SAS.000")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(metagroupRequest).GetStartTimestamp(), metagroupRequest.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(metagroupRequest).GetValues(), CreateValues(5, 21));
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(metagroupRequest).at(TStringBuf("SAS.000")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(metagroupRequest).at(TStringBuf("SAS.001")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);

        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(notFoundRequest).GetStartTimestamp(), notFoundRequest.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(notFoundRequest).GetValues(), TVector<TValue>{});
        UNIT_ASSERT(!visitor.StatusCodes.contains(notFoundRequest));
    }

    Y_UNIT_TEST(GroupAndMetagroupSameName) {
        THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(105),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("ASANDBOX"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        TYasmConf conf(GetYasmConf());
        THistoryAggregator aggregator(conf);
        aggregator.AddRequest(request);
        aggregator.AddMetagroup(TStringBuf("ASANDBOX"), {TStringBuf("ASANDBOX")});

        THistoryResponse response(
            request, TInstant::Seconds(100), CreateValues(1, 2), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        aggregator.Mul(response);

        TMockedHistoryVisitor visitor;
        aggregator.Visit(visitor);

        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(request).GetStartTimestamp(), request.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(request).GetValues(), CreateValues(1, 2));
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(request).at(TStringBuf("ASANDBOX")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
    }

    Y_UNIT_TEST(Overwrite) {
        TYasmConf conf(GetYasmConf());

        THistoryRequest fullHostRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(110),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        THistoryAggregator fullAggregator(conf);
        fullAggregator.AddRequest(fullHostRequest);
        THistoryResponse fullHostResponse(
            fullHostRequest, TInstant::Seconds(100), CreateValues(1, 2), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        fullAggregator.Mul(fullHostResponse);

        THistoryRequest realtimeHostRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(110),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };
        THistoryAggregator realtimeAggregator(conf);
        realtimeAggregator.AddRequest(realtimeHostRequest);
        THistoryResponse realtimeHostResponse(
            realtimeHostRequest, TInstant::Seconds(105), CreateValues(2, 2), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        realtimeAggregator.Mul(realtimeHostResponse);

        fullAggregator.Overwrite(realtimeAggregator);
        TMockedHistoryVisitor visitor;
        fullAggregator.Visit(visitor);
        TVector<TValue> expectedValues;
        expectedValues.emplace_back(1);
        expectedValues.emplace_back(2);
        expectedValues.emplace_back(2);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(fullHostRequest).GetStartTimestamp(), fullHostRequest.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(fullHostRequest).GetValues(), expectedValues);
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(fullHostRequest).at(TStringBuf("sas1-1234")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
    }

    Y_UNIT_TEST(MergeMultipleHosts) {
        THistoryRequest hostRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(300),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };

        TYasmConf conf(GetYasmConf());
        THistoryAggregator aggregator(conf);
        aggregator.AddRequest(hostRequest);

        THistoryResponse firstHostResponse(
            hostRequest, TInstant::Seconds(100), CreateValues(1, 21), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        THistoryResponse secondHostResponse(
            hostRequest, TInstant::Seconds(200), CreateValues(1, 21), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );

        aggregator.Mul(firstHostResponse);
        aggregator.Mul(secondHostResponse);

        TMockedHistoryVisitor visitor;
        aggregator.Visit(visitor);

        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(hostRequest).GetStartTimestamp(), hostRequest.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(hostRequest).GetValues(), CreateValues(1, 41));
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(hostRequest).at(TStringBuf("sas1-1234")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
    }

    Y_UNIT_TEST(DoNotMulMultipleHosts) {
        THistoryRequest hostRequest{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
        };

        TYasmConf conf(GetYasmConf());
        THistoryAggregator aggregator(conf);
        aggregator.AddRequest(hostRequest);

        THistoryResponse firstHostResponse(
            hostRequest, TInstant::Seconds(100), CreateValues(1, 21), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );
        THistoryResponse secondHostResponse(
            hostRequest, TInstant::Seconds(100), CreateValues(1, 21), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
        );

        aggregator.Mul(firstHostResponse);
        aggregator.Mul(secondHostResponse);

        TMockedHistoryVisitor visitor;
        aggregator.Visit(visitor);

        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(hostRequest).GetStartTimestamp(), hostRequest.Start);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Series.at(hostRequest).GetValues(), CreateValues(1, 21));
        UNIT_ASSERT_VALUES_EQUAL(visitor.StatusCodes.at(hostRequest).at(TStringBuf("sas1-1234")), EStatusCode::THistoryAggregatedSeries_EStatusCode_OK);
    }
}
