#include "preaggregates.h"

#include <util/memory/blob.h>

#include <library/cpp/yaml/as/tstring.h>

using namespace NZoom::NAggregators::NPrivate;
using namespace NZoom::NAggregators;
using namespace NZoom::NContainers;
using namespace NZoom::NRecord;
using namespace NTags;

namespace {
    class TCommonRulesVisitor: public TTagTreeVisitor {
    public:
        TCommonRulesVisitor(TInstanceKey instanceKey, const TVector<TCommonRuleMap>& rules)
            : InstanceKey(instanceKey)
            , Rules(rules)
            , Matched(false)
        {
        }

        void OnPath(TInternedTagNameSet, size_t position) override {
            const auto it(Rules[position].find(InstanceKey));
            if (!it.IsEnd()) {
                Matched = true;
                const auto& specificSelectors(it->second);
                for (const auto& [key, selector] : specificSelectors) {
                    if (key->Match(InstanceKey)) {
                        SpecificSignalSelector = &selector;
                        break;
                    }
                }
            }
        }

        bool IsMatched() const noexcept {
            return Matched;
        }

        TSelectorPtr GetSpecificSignalSelector() const noexcept {
            return SpecificSignalSelector;
        };

    private:
        const TInstanceKey InstanceKey;
        const TVector<TCommonRuleMap>& Rules;
        bool Matched;
        TSelectorPtr SpecificSignalSelector = nullptr;
    };

    class TAggregationRulesVisitor: public TTagTreeVisitor {
    public:
        TAggregationRulesVisitor(TInstanceKey instanceKey, const TVector<TInstanceRuleMap>& rules, TTagNames& tagNames)
            : InstanceKey(instanceKey)
            , Rules(rules)
            , TagNames(tagNames)
        {
        }

        void OnPath(TInternedTagNameSet, size_t position) override {
            const auto it = Rules[position].find(InstanceKey);
            if (!it.IsEnd()) {
                TagNames.insert(it->second.begin(), it->second.end());
            }
        }

        const TTagNames& GetTagNames() const noexcept {
            return TagNames;
        }

    private:
        const TInstanceKey InstanceKey;
        const TVector<TInstanceRuleMap>& Rules;
        TTagNames& TagNames;
    };
}

TRuleKey::TRuleKey(const NTags::TRequestKey& requestKey)
    : TagNameSet(requestKey.GetTagNameSet())
    , TagValueHash(requestKey.GetValueHash())
{
}

size_t TRuleKey::Hash() const noexcept {
    return TagValueHash;
}

bool TRuleKey::operator==(const TRuleKey& other) const noexcept {
    return TagValueHash == other.TagValueHash;
}

bool TRuleKey::operator==(NTags::TInstanceKey other) const noexcept {
    return TagValueHash == other.GetValueHash(TagNameSet);
}

bool TRuleKey::operator==(const NTags::TRequestKey& other) const noexcept {
    return TagValueHash == other.GetValueHash();
}

void ITagAgentContainerCallback::SetObjectsCount(const size_t /* count */) {
}

TInstanceRuleMap& TAggregationRules::GetOrCreateRuleMap(const NTags::TRequestKey& requestKey) {
    const auto& vertex(TagTree.Create(requestKey));
    if (vertex.IsLeaf()) {
        return CustomRules[vertex.GetPosition()];
    } else {
        vertex.SetPosition(CustomRules.size());
        return CustomRules.emplace_back(0, TRuleKeyHash(requestKey.GetTagNameSet()));
    }
}

TTagNames& TAggregationRules::GetOrCreateRules(const NTags::TRequestKey& requestKey) {
    TInstanceRuleMap::insert_ctx context;
    auto& instanceRuleMap(GetOrCreateRuleMap(requestKey));
    const auto it(instanceRuleMap.find(requestKey, context));
    if (it != instanceRuleMap.end()) {
        return it->second;
    } else {
        return instanceRuleMap.emplace_direct(context, requestKey, 0)->second;
    }
}

void TAggregationRules::AddCustomRule(TStringBuf requestKey, const TVector<TStringBuf>& tagNames) {
    AddCustomRule(NTags::TRequestKey::FromString(requestKey), tagNames);
}

void TAggregationRules::AddCustomRule(const NTags::TRequestKey& requestKey, const TVector<TStringBuf>& tagNames) {
    if (!requestKey.HasOnlyEqualMatcher()) {
        ythrow yexception() << "request key " << requestKey << " can't be added (only 'equal' matchers supported)";
    }
    GetOrCreateRules(requestKey).emplace(TTagNameSet(tagNames));
}

void TAggregationRules::AddDefaultRule(const TVector<TStringBuf>& tagNames) {
    DefaultRules.emplace(TTagNameSet(tagNames));
}

TTagNames TAggregationRules::GetRules(TInstanceKey instanceKey) const {
    TTagNames tagNames;
    for (const auto& key : instanceKey.CartesianProduct()) {
        TAggregationRulesVisitor visitor(key, CustomRules, tagNames);
        TagTree.Find(key, visitor);
    }

    if (tagNames.empty()) {
        return DefaultRules;
    }
    return tagNames;
}

void NZoom::NAggregators::FillRules(TCommonRules& commonRules, TAggregationRules& aggregationRules, TStringBuf path) {
    FillRules(commonRules, aggregationRules, TFsPath(path));
}

void NZoom::NAggregators::FillRules(TCommonRules& commonRules, TAggregationRules& aggregationRules, const TFsPath& path) {
    const auto blob(TBlob::FromFile(path));
    YAML::Node yamlRules = YAML::Load(static_cast<std::string>(TStringBuf(blob.AsCharPtr(), blob.Size())));

    {
        auto aggregationRulesRaw = yamlRules["aggregation_rules"];

        auto customRules = aggregationRulesRaw["custom"];
        for (auto part : customRules) {
            TVector<TString> tagsRaw;
            TVector<TStringBuf> tags;
            for (YAML::Node tag : part["tags"]) {
                tagsRaw.emplace_back(tag.as<TString>());
                tags.emplace_back(tagsRaw.back());
            }
            aggregationRules.AddCustomRule(part["request_key"].as<TString>(), tags);
        }

        for (auto part : aggregationRulesRaw["default"]) {
            TVector<TString> tagsRaw;
            TVector<TStringBuf> tags;
            for (YAML::Node tag : part["tags"]) {
                tagsRaw.emplace_back(tag.as<TString>());
                tags.emplace_back(tagsRaw.back());
            }
            aggregationRules.AddDefaultRule(tags);
        }
    }

    {
        for (YAML::Node rule : yamlRules["common_rules"]) {
            if (rule["selector"]) {
                commonRules.AddRule(
                        rule["request_key"].as<TString>(),
                        rule["selector"].as<TString>());
            } else {
                commonRules.AddRule(rule["request_key"].as<TString>());
            }
        }
    }
}

TCommonSignalRules& TCommonRules::GetOrCreateRules(const TRequestKey& requestKey) {
    TCommonRuleMap::insert_ctx context;
    auto& commonRuleMap(GetOrCreateRuleMap(requestKey));
    const auto it(commonRuleMap.find(requestKey, context));
    if (it != commonRuleMap.end()) {
        return it->second;
    } else {
        return commonRuleMap.emplace_direct(context, requestKey, 0)->second;
    }
}

TCommonRuleMap& TCommonRules::GetOrCreateRuleMap(const NTags::TRequestKey& requestKey) {
    const auto& vertex(TagTree.Create(requestKey));
    if (vertex.IsLeaf()) {
        return Rules[vertex.GetPosition()];
    } else {
        vertex.SetPosition(Rules.size());
        return Rules.emplace_back(0, TRuleKeyHash(requestKey.GetTagNameSet()));
    }
}

void TCommonRules::AddRule(TStringBuf requestKeyRaw, TStringBuf selectorRaw) {
    auto requestKey(NTags::TRequestKey::FromString(requestKeyRaw));
    if (!requestKey.HasOnlyEqualMatcher()) {
        ythrow yexception() << "request key " << requestKey << " can't be added (only 'equal' matchers supported)";
    }

    auto& rules(GetOrCreateRules(requestKey));
    if (selectorRaw != "signal=*") {
        auto selector = NSolomon::ParseSelector(selectorRaw);
        if (selector.Key() != "signal") {
            ythrow yexception() << "request key " << requestKey
                                << " can't be added (only 'signal' selectors supported"
                                << ", but '" << selectorRaw << "' given)";
        }

        rules.emplace_back(MakeHolder<NTags::TRequestKey>(std::move(requestKey)), selector);
    }
}

TSelectorPtr TCommonRules::Apply(NTags::TInstanceKey instanceKey) const {
    for (const auto& key : instanceKey.CartesianProduct()) {
        TCommonRulesVisitor visitor(key, Rules);
        TagTree.Find(key, visitor);
        if (visitor.IsMatched()) {
            const auto specificSignalSelector = visitor.GetSpecificSignalSelector();
            if (specificSignalSelector) {
                return specificSignalSelector;
            } else {
                return &DefaultSignalSelector;
            }
        }
    }
    return nullptr;
}

class TConfigurableAggregator::TCallback final: public ITagRecordCallback {
public:
    TCallback(TConfigurableAggregator& aggregator)
        : Conf(aggregator.Conf)
        , Rules(aggregator.Rules)
        , InstanceMap(aggregator.Storage)
    {
    }

    void OnTagRecord(TInstanceKey instanceKey, const TRecord& record) override {
        if (!instanceKey.GetAggregated().empty()) {
            return;
        }

        THashSet<TInstanceKey> addedToPreaggregates;
        for (const auto rule : Rules.GetRules(instanceKey)) {
            auto aggregatedKey = instanceKey.AggregateBy(rule);
            if (addedToPreaggregates.insert(aggregatedKey).second) {
                GetContainer(aggregatedKey).Mul(record, MetricManager);
            }
        }
    }

private:
    NZoom::NContainers::TGroupContainer& GetContainer(TInstanceKey key) {
        TInstanceMap::insert_ctx context;
        const auto it(InstanceMap.find(key, context));
        if (it != InstanceMap.end()) {
            return it->second;
        }
        return InstanceMap.emplace_direct(context, key, Conf.GetTypeConf(key.GetItype(), false))->second;
    }

    const NZoom::NYasmConf::TYasmConf& Conf;
    const TAggregationRules& Rules;
    TInstanceMap& InstanceMap;
    NZoom::NValue::TMetricManager MetricManager;
};

TConfigurableAggregator::TConfigurableAggregator(const NZoom::NYasmConf::TYasmConf& conf, const TAggregationRules& rules)
    : Conf(conf)
    , Rules(rules)
{
}

void TConfigurableAggregator::Mul(const NZoom::NRecord::TTaggedRecord& taggedRecord) {
    TCallback callback(*this);
    taggedRecord.Process(callback);
}

size_t TConfigurableAggregator::Len() const {
    size_t elements = 0;
    for (const auto & [_, container ] : Storage) {
        elements += container.Len();
    }
    return elements;
}

void TConfigurableAggregator::Clean() {
    Clean(TInstant::Now());
}

void TConfigurableAggregator::Clean(const TInstant explicitNow) {
    for (auto instanceIt = Storage.begin(); instanceIt != Storage.end();) {
        auto prevInstanceIt = instanceIt;
        ++instanceIt;

        if (prevInstanceIt->second.Len()) {
            prevInstanceIt->second.Clean(explicitNow);
        }

        if (!prevInstanceIt->second.Len()) {
            Storage.erase(prevInstanceIt);
        }
    }
}

void TConfigurableAggregator::Process(ITagAgentContainerCallback& callback) const {
    callback.SetObjectsCount(Storage.size());
    for (const auto & [ instance, container ] : Storage) {
        callback.OnTagContainer(instance, container);
    }
}
