#pragma once

#include "tag_tree.h"

#include <infra/yasm/common/labels/tags/request_key.h>
#include <infra/yasm/zoom/components/containers/group.h>
#include <infra/yasm/zoom/components/record/record.h>
#include <infra/yasm/common/labels/host/host.h>

#include <solomon/libs/cpp/selectors/selectors.h>

#include <util/folder/path.h>

namespace NZoom::NAggregators {
    namespace NPrivate {
        using TTagValueHash = size_t;

        using TInstanceMap = THashMap<NTags::TInstanceKey, NZoom::NContainers::TGroupContainer>;

        class TRuleKey {
        public:
            TRuleKey(const NTags::TRequestKey& requestKey);

            size_t Hash() const noexcept;

            bool operator==(const TRuleKey& other) const noexcept;
            bool operator==(NTags::TInstanceKey other) const noexcept;
            bool operator==(const NTags::TRequestKey& other) const noexcept;

        private:
            NTags::TInternedTagNameSet TagNameSet;
            size_t TagValueHash;
        };

        struct TRuleKeyHash {
            TRuleKeyHash(NTags::TInternedTagNameSet tagNameSet)
                : TagNameSet(tagNameSet)
            {
            }

            inline size_t operator()(const TRuleKey& container) const noexcept {
                return container.Hash();
            }

            inline size_t operator()(NTags::TInstanceKey key) const noexcept {
                return key.GetValueHash(TagNameSet);
            }

            inline size_t operator()(const NTags::TRequestKey& key) const noexcept {
                return key.GetValueHash();
            }

            NTags::TInternedTagNameSet TagNameSet;
        };

        struct TRuleKeyEqualTo {
            inline bool operator()(const TRuleKey& lhs, const TRuleKey& rhs) const noexcept {
                return lhs == rhs;
            }

            inline bool operator()(const TRuleKey& lhs, NTags::TInstanceKey rhs) const noexcept {
                return lhs == rhs;
            }

            inline bool operator()(const TRuleKey& lhs, const NTags::TRequestKey& rhs) const noexcept {
                return lhs == rhs;
            }
        };

        using TTagNames = THashSet<NTags::TInternedTagNameSet>;
        using TInstanceRuleMap = THashMap<TRuleKey, TTagNames, TRuleKeyHash, TRuleKeyEqualTo>;

        using TRequestKeyPtr = THolder<NTags::TRequestKey>;
        using TCommonSignalRules = TVector<std::pair<TRequestKeyPtr, NSolomon::TSelector>>;
        using TCommonRuleMap = THashMap<TRuleKey, TCommonSignalRules, TRuleKeyHash, TRuleKeyEqualTo>;
    }

    using TSelectorPtr = const NSolomon::TSelector*;

    class TCommonRules {

    public:
        void AddRule(TStringBuf requestKey, TStringBuf selector = "signal=*");

        TSelectorPtr Apply(NTags::TInstanceKey instanceKey) const;

    private:
        TTagTree TagTree;
        TVector<NPrivate::TCommonRuleMap> Rules;
        const NSolomon::TSelector DefaultSignalSelector = NSolomon::ParseSelector("signal=*");

        NPrivate::TCommonRuleMap& GetOrCreateRuleMap(const NTags::TRequestKey& requestKey);
        NPrivate::TCommonSignalRules& GetOrCreateRules(const NTags::TRequestKey& requestKey);
    };

    struct ITagAgentContainerCallback {
        virtual ~ITagAgentContainerCallback() = default;

        virtual void SetObjectsCount(const size_t count);
        virtual void OnTagContainer(NTags::TInstanceKey key, const NZoom::NContainers::TGroupContainer& container) = 0;
    };

    class TAggregationRules {
    public:
        void AddCustomRule(TStringBuf requestKey, const TVector<TStringBuf>& tagNames);
        void AddCustomRule(const NTags::TRequestKey& requestKey, const TVector<TStringBuf>& tagNames);
        void AddDefaultRule(const TVector<TStringBuf>& tagNames);

        NPrivate::TTagNames GetRules(NTags::TInstanceKey instanceKey) const;

    private:
        TTagTree TagTree;
        TVector<NPrivate::TInstanceRuleMap> CustomRules;
        NPrivate::TTagNames DefaultRules;

        NPrivate::TInstanceRuleMap& GetOrCreateRuleMap(const NTags::TRequestKey& requestKey);
        NPrivate::TTagNames& GetOrCreateRules(const NTags::TRequestKey& requestKey);
    };

    void FillRules(TCommonRules& commonRules, TAggregationRules& aggregationRules, const TFsPath& path);
    void FillRules(TCommonRules& commonRules, TAggregationRules& aggregationRules, TStringBuf path);

    class TConfigurableAggregator {
    public:
        TConfigurableAggregator(const NZoom::NYasmConf::TYasmConf& conf, const TAggregationRules& rules);

        void Mul(const NZoom::NRecord::TTaggedRecord& taggedRecord);

        size_t Len() const;
        void Clean();
        void Clean(const TInstant explicitNow); // Debugging hook
        void Process(ITagAgentContainerCallback& callback) const;

    private:
        class TCallback;

        const NZoom::NYasmConf::TYasmConf& Conf;
        const TAggregationRules& Rules;

        NPrivate::TInstanceMap Storage;
    };
}
