#include "preaggregates.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/env.h>

using namespace NTags;
using namespace NZoom::NAggregators;
using namespace NZoom::NYasmConf;
using namespace NZoom::NSignal;
using namespace NZoom::NHost;
using namespace NZoom::NValue;
using namespace NZoom::NRecord;
using namespace NZoom::NContainers;

namespace {
    TYasmConf GetYasmConf() {
        return TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {}, \"patterns\": {}, \"periods\": {}}}}"));
    }

    struct TSignalVisitorMock: public ISignalValueCallback {
        void OnSignalValue(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value) override {
            Y_VERIFY(Signals.emplace(name, value).first);
        }

        THashMap<TSignalName, TValue> Signals;
    };

    struct THostTagVisitorMock final: public ITagAgentContainerCallback {
        void SetObjectsCount(const size_t count) override {
            Instances.reserve(count);
        }

        void OnTagContainer(TInstanceKey key, const TGroupContainer& container) override {
            container.Process(Instances[key]);
        }

        THashMap<TInstanceKey, TSignalVisitorMock> Instances;
    };
}

Y_UNIT_TEST_SUITE(TAggregateByPipelineTest) {
    TTaggedRecord CreateRecord(TInstanceKey instanceKey) {
        TVector<std::pair<TSignalName, TValue>> signalVec;
        signalVec.emplace_back(TStringBuf("portoinst-rss_gb_summ"), TValue(10));
        TRecord record(std::move(signalVec));

        TVector<std::pair<TInstanceKey, TRecord>> tagVec;
        tagVec.emplace_back(instanceKey, std::move(record));
        return {std::move(tagVec)};
    }

    Y_UNIT_TEST(ContainerCleanup) {
        TYasmConf yasmConf(GetYasmConf());

        TAggregationRules rules;
        rules.AddDefaultRule(TVector<TStringBuf>{"tier"});
        rules.AddDefaultRule(TVector<TStringBuf>{"tier", "prj", "ctype", "geo"});

        TInstant now(TInstant::Now());
        TTaggedRecord record(CreateRecord(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub;tier=none")));

        TConfigurableAggregator aggregator(yasmConf, rules);
        aggregator.Mul(record);
        UNIT_ASSERT_VALUES_EQUAL(aggregator.Len(), 2);

        aggregator.Clean(now);
        UNIT_ASSERT_VALUES_EQUAL(aggregator.Len(), 2);

        aggregator.Clean(now + TDuration::Seconds(60));
        UNIT_ASSERT_VALUES_EQUAL(aggregator.Len(), 0);
    }

    Y_UNIT_TEST(NoMatch) {
        TYasmConf yasmConf(GetYasmConf());

        TAggregationRules rules;
        rules.AddCustomRule(TRequestKey::FromString("itype=wrong"), TVector<TStringBuf>{"tier"});

        TConfigurableAggregator aggregator(yasmConf, rules);
        aggregator.Mul(
            CreateRecord(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub;tier=none"))
        );

        UNIT_ASSERT_VALUES_EQUAL(aggregator.Len(), 0);
    }

    Y_UNIT_TEST(NoAggregated) {
        TYasmConf yasmConf(GetYasmConf());

        TAggregationRules rules;
        rules.AddCustomRule(TRequestKey::FromString("itype=balancer"), TVector<TStringBuf>{"tier"});

        TConfigurableAggregator aggregator(yasmConf, rules);
        aggregator.Mul(
            CreateRecord(TInstanceKey::FromNamed("balancer||geo,ctype,prj,tier"))
        );

        UNIT_ASSERT_VALUES_EQUAL(aggregator.Len(), 0);
    }

    Y_UNIT_TEST(NoComplexRules) {
        TYasmConf yasmConf(GetYasmConf());

        TAggregationRules rules;
        UNIT_ASSERT_EXCEPTION(rules.AddCustomRule(
            TRequestKey::FromString("itype=balancer;ctype=prod,prestable"),
            TVector<TStringBuf>{"tier"}
        ), yexception);
    }

    TValue& GetValueFromVisitor(THostTagVisitorMock & visitor, TStringBuf key, TStringBuf signal) {
        return visitor.Instances.at(TInstanceKey::FromNamed(key)).Signals.at(signal);
    }

    Y_UNIT_TEST(MultipleMatch) {
        TYasmConf yasmConf(GetYasmConf());

        TAggregationRules rules;
        rules.AddCustomRule(TRequestKey::FromString("itype=a"), TVector<TStringBuf>{"b"});
        rules.AddCustomRule(TRequestKey::FromString("itype=a"), TVector<TStringBuf>{"b", "c"});
        rules.AddCustomRule(TRequestKey::FromString("itype=a"), TVector<TStringBuf>{"b", "c", "d"});

        TConfigurableAggregator aggregator(yasmConf, rules);
        aggregator.Mul(
            CreateRecord(TInstanceKey::FromNamed("a|b=b1"))
        );

        THostTagVisitorMock visitor;
        aggregator.Process(visitor);
        UNIT_ASSERT_VALUES_EQUAL(
            GetValueFromVisitor(visitor, "a||b", "portoinst-rss_gb_summ"),
            TValue(10)
        );
    }

    Y_UNIT_TEST(Rules) {
        TYasmConf yasmConf(GetYasmConf());

        TAggregationRules rules;
        rules.AddCustomRule(TRequestKey::FromString("itype=balancer"), TVector<TStringBuf>{"prj"});
        rules.AddCustomRule(TRequestKey::FromString("itype=balancer"), TVector<TStringBuf>{"ctype", "prj", "geo", "tier"});
        rules.AddDefaultRule(TVector<TStringBuf>{"tier"});
        rules.AddDefaultRule(TVector<TStringBuf>{"ctype", "prj", "geo", "tier"});

        TConfigurableAggregator aggregator(yasmConf, rules);
        aggregator.Mul(
            CreateRecord(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub;tier=none"))
        );
        aggregator.Mul(
            CreateRecord(TInstanceKey::FromNamed("upper|geo=sas;ctype=prod;prj=rkub;tier=none"))
        );
        aggregator.Mul(
            CreateRecord(TInstanceKey::FromNamed("upper|geo=sas;ctype=prod;prj=rkub;tier=none"))
        );

        THostTagVisitorMock visitor;
        aggregator.Process(visitor);
        UNIT_ASSERT_VALUES_EQUAL(
            GetValueFromVisitor(visitor, "upper|geo=sas;ctype=prod;prj=rkub|tier", "portoinst-rss_gb_summ"),
            TValue(20)
        );
        UNIT_ASSERT_VALUES_EQUAL(
            GetValueFromVisitor(visitor, "upper||geo,ctype,prj,tier", "portoinst-rss_gb_summ"),
            TValue(20)
        );
        UNIT_ASSERT_VALUES_EQUAL(
            GetValueFromVisitor(visitor, "balancer|geo=sas;ctype=prod;tier=none|prj", "portoinst-rss_gb_summ"),
            TValue(10)
        );
        UNIT_ASSERT_VALUES_EQUAL(
            GetValueFromVisitor(visitor, "balancer||geo,ctype,prj,tier", "portoinst-rss_gb_summ"),
            TValue(10)
        );
        UNIT_ASSERT(!visitor.Instances.contains(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub|tier")));
    }

    Y_UNIT_TEST(TCommonRules) {
        TCommonRules commonRules;
        commonRules.AddRule("itype=balancer");
        commonRules.AddRule("itype=qloud;prj=something");

        UNIT_ASSERT(commonRules.Apply(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub|tier")));
        UNIT_ASSERT(!commonRules.Apply(TInstanceKey::FromNamed("apphost|geo=sas;ctype=prod;prj=rkub|tier")));
        UNIT_ASSERT(commonRules.Apply(TInstanceKey::FromNamed("qloud|geo=sas;ctype=prod;prj=something;tier=none")));
        UNIT_ASSERT(!commonRules.Apply(TInstanceKey::FromNamed("qloud|geo=sas;ctype=prod;prj=rkub;tier=none")));

        auto defaultAny1(commonRules.Apply(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub|tier")));
        UNIT_ASSERT(defaultAny1->Match("sdfasdfasdf"));

        auto defaultAny2(commonRules.Apply(TInstanceKey::FromNamed("qloud|geo=sas;ctype=prod;prj=something;tier=none")));
        UNIT_ASSERT(defaultAny2->Match("qwerty"));
    }

    Y_UNIT_TEST(TCommonRulesWithSignalSelector) {
        TCommonRules commonRules;
        commonRules.AddRule("itype=balancer", "signal=*");
        commonRules.AddRule("itype=balancer;prj=zxc", "signal=mem-cache_thhh|mem-free*");
        commonRules.AddRule("itype=qloud;prj=something", "signal=mem-*");

        auto any1(commonRules.Apply(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub|tier")));
        UNIT_ASSERT(any1->Match("cpu-id"));
        UNIT_ASSERT(any1->Match("mem-free"));
        UNIT_ASSERT(any1->Match("mem-cache"));
        UNIT_ASSERT(any1->Match("mem-cache_thhh"));
        UNIT_ASSERT(any1->Match("mem-cache_tvvv"));
        auto any2(commonRules.Apply(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub|tier")));
        UNIT_ASSERT(any2->Match("cpu-id"));
        UNIT_ASSERT(any2->Match("mem-cache"));
        UNIT_ASSERT(any2->Match("mem-free"));
        UNIT_ASSERT(any2->Match("mem-free_thhh"));
        UNIT_ASSERT(any2->Match("mem-free_tvvv"));

        auto pipe1(commonRules.Apply(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=zxc|tier")));
        UNIT_ASSERT(!pipe1->Match("cpu-id"));
        UNIT_ASSERT(!pipe1->Match("mem-cache"));
        UNIT_ASSERT(pipe1->Match("mem-free"));
        UNIT_ASSERT(pipe1->Match("mem-free_thhh"));
        UNIT_ASSERT(pipe1->Match("mem-free_tvvv"));
        auto pipe2(commonRules.Apply(TInstanceKey::FromNamed("balancer|geo=man;ctype=prod;prj=zxc|tier")));
        UNIT_ASSERT(!pipe2->Match("cpu-id"));
        UNIT_ASSERT(pipe2->Match("mem-free"));
        UNIT_ASSERT(!pipe2->Match("mem-cache"));
        UNIT_ASSERT(pipe2->Match("mem-cache_thhh"));
        UNIT_ASSERT(!pipe2->Match("mem-cache_tvvv"));

        auto mem1(commonRules.Apply(TInstanceKey::FromNamed("qloud|geo=sas;ctype=prod;prj=something;tier=none")));
        UNIT_ASSERT(!mem1->Match("cpu-id"));
        UNIT_ASSERT(mem1->Match("mem-free"));
        UNIT_ASSERT(mem1->Match("mem-cache"));
        UNIT_ASSERT(mem1->Match("mem-cache_thhh"));
        UNIT_ASSERT(mem1->Match("mem-cache_tvvv"));
        auto mem2(commonRules.Apply(TInstanceKey::FromNamed("qloud|geo=sas;ctype=prestable;prj=something;tier=none")));
        UNIT_ASSERT(!mem2->Match("cpu-id"));
        UNIT_ASSERT(mem2->Match("mem-cache"));
        UNIT_ASSERT(mem2->Match("mem-free"));
        UNIT_ASSERT(mem2->Match("mem-free_thhh"));
        UNIT_ASSERT(mem2->Match("mem-free_tvvv"));
    }

    Y_UNIT_TEST(TTestRulesFromFile) {
        TCommonRules commonRules;
        TAggregationRules aggregationRules;
        TFsPath rulesPath = JoinPaths(TStringBuf(ArcadiaSourceRoot()), TStringBuf("infra/yasm/zoom/components/aggregators/ut/rules.conf"));
        FillRules(commonRules, aggregationRules, rulesPath);

        TVector<std::pair<TInstanceKey, THashSet<TInstanceKey>>> testCases{
            {TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub;tier=none"), {
                TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;tier=none|prj"),
                TInstanceKey::FromNamed("balancer||geo,ctype,tier,prj")
            }},
            {TInstanceKey::FromNamed("yasm|geo=sas;ctype=prod;prj=rkub;tier=none"), {
                TInstanceKey::FromNamed("yasm||geo,ctype,tier,prj")
            }}
        };

        for (const auto & [ key, expected ] : testCases) {
            THashSet<TInstanceKey> result;
            auto keyRules = aggregationRules.GetRules(key);
            for (const auto& rule : keyRules ) {
                result.insert(key.AggregateBy(rule));
            }
            UNIT_ASSERT_VALUES_EQUAL(keyRules.size(), expected.size());
            UNIT_ASSERT_VALUES_EQUAL(result, expected);
        }

        UNIT_ASSERT(commonRules.Apply(TInstanceKey::FromNamed("qloud")));
        UNIT_ASSERT(commonRules.Apply(TInstanceKey::FromNamed("balancer|prj=none;tier=none")));
        UNIT_ASSERT(commonRules.Apply(TInstanceKey::FromNamed("balancer|prj=none;tier=none")));

        UNIT_ASSERT(!commonRules.Apply(TInstanceKey::FromNamed("balancer|prj=none;tier=some")));
        UNIT_ASSERT(!commonRules.Apply(TInstanceKey::FromNamed("balancer|prj=none")));
        UNIT_ASSERT(!commonRules.Apply(TInstanceKey::FromNamed("apphost")));

        auto any1(commonRules.Apply(TInstanceKey::FromNamed("qloud")));
        UNIT_ASSERT(any1->Match("cpu-id"));
        UNIT_ASSERT(any1->Match("cpu-id_hgram"));
        UNIT_ASSERT(any1->Match("mem-free"));
        UNIT_ASSERT(any1->Match("mem-cache"));

        auto any2(commonRules.Apply(TInstanceKey::FromNamed("balancer|prj=none,abc;tier=none")));
        UNIT_ASSERT(any2->Match("cpu-id"));
        UNIT_ASSERT(any2->Match("cpu-id_hgram"));
        UNIT_ASSERT(any2->Match("mem-free"));
        UNIT_ASSERT(any2->Match("mem-cache"));

        auto cpu(commonRules.Apply(TInstanceKey::FromNamed("cpu")));
        UNIT_ASSERT(cpu->Match("cpu-id"));
        UNIT_ASSERT(cpu->Match("cpu-id_hgram"));
        UNIT_ASSERT(!cpu->Match("mem-free"));
        UNIT_ASSERT(!cpu->Match("mem-cache"));

        auto pipe(commonRules.Apply(TInstanceKey::FromNamed("pipe")));
        UNIT_ASSERT(pipe->Match("cpu-id"));
        UNIT_ASSERT(!pipe->Match("cpu-id_hgram"));
        UNIT_ASSERT(pipe->Match("mem-free"));
        UNIT_ASSERT(pipe->Match("mem-cache"));

        auto negate(commonRules.Apply(TInstanceKey::FromNamed("negate")));
        UNIT_ASSERT(!negate->Match("cpu-id"));
        UNIT_ASSERT(negate->Match("cpu-id_hgram"));
        UNIT_ASSERT(!negate->Match("mem-free"));
        UNIT_ASSERT(!negate->Match("mem-cache"));
    }

    Y_UNIT_TEST(TTestAggrRulesConf) {
        TYasmConf yasmConf(GetYasmConf());
        TCommonRules commonRules;
        TAggregationRules aggregationRules;

        TFsPath rulesPath = JoinPaths(TStringBuf(ArcadiaSourceRoot()), TStringBuf("infra/yasm/zoom/components/aggregators/ut/rules.conf"));
        FillRules(commonRules, aggregationRules, rulesPath);

        TConfigurableAggregator aggregator(yasmConf, aggregationRules);
        aggregator.Mul(
            CreateRecord(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub;tier=none"))
        );

        THostTagVisitorMock visitor;
        aggregator.Process(visitor);
        UNIT_ASSERT_VALUES_EQUAL(
            GetValueFromVisitor(visitor, "balancer|geo=sas;ctype=prod;tier=none|prj", "portoinst-rss_gb_summ"),
            TValue(10)
        );
        UNIT_ASSERT_VALUES_EQUAL(
            GetValueFromVisitor(visitor, "balancer||geo,ctype,prj,tier", "portoinst-rss_gb_summ"),
            TValue(10)
        );
        UNIT_ASSERT(!visitor.Instances.contains(TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub|tier")));
    }

    Y_UNIT_TEST(TTestTAggregationRules) {
        TAggregationRules rules;

        rules.AddCustomRule(TRequestKey::FromString("itype=balancer"), TVector<TStringBuf>{"prj"});
        rules.AddCustomRule(TRequestKey::FromString("itype=balancer"), TVector<TStringBuf>{"ctype", "prj", "geo", "tier"});
        rules.AddCustomRule(TRequestKey::FromString("itype=balancer"), TVector<TStringBuf>{"ctype", "prj", "geo", "tier"});

        rules.AddDefaultRule(TVector<TStringBuf>{"tier"});
        rules.AddDefaultRule(TVector<TStringBuf>{"ctype", "geo", "tier"});

        TVector<std::pair<TInstanceKey, THashSet<TInstanceKey>>> testCases{
            {TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;prj=rkub;tier=none"), {
                TInstanceKey::FromNamed("balancer|geo=sas;ctype=prod;tier=none|prj"),
                TInstanceKey::FromNamed("balancer||geo,ctype,tier,prj")
            }},
            {TInstanceKey::FromNamed("yasm|geo=sas;ctype=prod;prj=rkub;tier=none"), {
                TInstanceKey::FromNamed("yasm|geo=sas;ctype=prod;prj=rkub|tier"),
                TInstanceKey::FromNamed("yasm|prj=rkub|geo,ctype,tier")
            }}
        };

        for (const auto & [ key, expected ] : testCases) {
            THashSet<TInstanceKey> result;
            auto tagRules = rules.GetRules(key);
            for (const auto& rule : tagRules) {
                result.insert(key.AggregateBy(rule));
            }
            UNIT_ASSERT_VALUES_EQUAL(tagRules.size(), expected.size());
            UNIT_ASSERT_VALUES_EQUAL(result, expected);
        }
    }
}
