#include "subscription_filter.h"

#include <util/digest/multi.h>
#include <util/generic/xrange.h>

using namespace NTags;
using namespace NZoom::NAggregators;
using namespace NZoom::NSignal;

namespace {
    class TMarkVisitor : public TTagTreeVisitor {
    public:
        TMarkVisitor(TInstanceKey instanceKey, const TVector<TSignalName>& signals,
                     TDynBitMap& matches, const TSubscriptionFilter::TSeriesMask& seriesMask)
            : InstanceKey(instanceKey)
            , Signals(signals)
            , Matches(matches)
            , SeriesMask(seriesMask)
        {
        }

        void OnPath(TInternedTagNameSet tagNameSet, size_t) override {
            const auto tagHash(InstanceKey.GetValueHash(tagNameSet));
            for (const auto idx : xrange(Signals.size())) {
                if (SeriesMask.Get(CombineHashes(Signals[idx].Hash(), tagHash) % SeriesMask.Size())) {
                    Matches.Set(idx);
                }
            }
        }

    private:
        TInstanceKey InstanceKey;
        const TVector<TSignalName>& Signals;
        TDynBitMap& Matches;
        const TSubscriptionFilter::TSeriesMask& SeriesMask;
    };
}

void TSubscriptionFilter::Add(const TRequestKey& requestKey, const TVector<TSignalName>& signals) noexcept {
    const auto& vertex(TagTree.Create(requestKey));
    vertex.MarkAsLeaf();
    const auto tagHash(requestKey.GetValueHash());
    for (const auto& signal : signals) {
        SeriesMask.Set(CombineHashes(signal.Hash(), tagHash) % SeriesMask.Size());
    }
}

TDynBitMap TSubscriptionFilter::Match(TInstanceKey instanceKey, const TVector<TSignalName>& signals) const noexcept {
    TDynBitMap result;
    result.Reserve(signals.size());
    for (const auto& key : instanceKey.CartesianProduct()) {
        TMarkVisitor visitor(key, signals, result, SeriesMask);
        TagTree.Find(key, visitor);
    }
    return result;
}

void TSubscriptionFilter::Clear() {
    TagTree.Clear();
    SeriesMask.Clear();
}
