#include "tag_tree.h"

using namespace NZoom::NAggregators;
using namespace NTags;

bool TTagVertexLess::operator()(const TTagVertex& lhs, const TTagVertex& rhs) const noexcept {
    return lhs.GetName() < rhs.GetName();
}

bool TTagVertexLess::operator()(const TTagVertex& lhs, const TTagName& rhs) const noexcept {
    return lhs.GetName() < rhs;
}

bool TTagVertexLess::operator()(const TTagName& lhs, const TTagVertex& rhs) const noexcept {
    return lhs < rhs.GetName();
}

void TTagVertex::Visit(TTagTreeVisitor& visitor) const {
    if (IsLeaf()) {
        visitor.OnPath(GetPath(), GetPosition());
    }
}

void TTagVertex::Clear() const {
    Position.Clear();
    Vertices.clear();
}

void TTagVertex::Out(IOutputStream& stream) const {
    stream.Write(TStringBuf("("));
    stream.Write(Name.GetName());
    stream.Write(TStringBuf(" "));
    if (IsLeaf()) {
        stream.Write(ToString(GetPosition()));
    } else {
        stream.Write(TStringBuf("null"));
    }
    stream.Write(TStringBuf(" ("));
    bool first = true;
    for (const auto& vertex : Vertices) {
        if (!first) {
            stream.Write(TStringBuf(" "));
        }
        vertex.Out(stream);
        first = false;
    }
    stream.Write(TStringBuf(")"));
    stream.Write(TStringBuf(")"));
}

TTagTree::TTagTree()
    : Root(TTagName("itype"), TVector<TTagName>())
{
}

const TTagVertex& TTagTree::Create(const TRequestKey& requestKey) {
    const TTagVertex* current = &Root;
    TVector<NTags::TTagName> currentPath;
    for (const auto& tagName : requestKey.GetTagNameSet().GetOrdered()) {
        currentPath.emplace_back(tagName);
        auto it = current->Vertices.lower_bound(tagName);
        if (it == current->Vertices.end() || it->GetName() != tagName) {
            it = current->Vertices.emplace_hint(it, TTagVertex(tagName, currentPath));
        }
        current = &(*it);
    }
    return *current;
}

void TTagTree::Find(TInstanceKey instanceKey, TTagTreeVisitor& visitor) const {
    const auto& tags(instanceKey.GetTagNameSet());
    Root.Visit(visitor);
    StepInto(Root, tags.GetOrdered().begin(), tags.GetOrdered().end(), visitor);
}

void TTagTree::StepInto(const TTagVertex& vertex, TTagIterator tagIterator, const TTagIterator tagEnd, TTagTreeVisitor& visitor) const {
    TTagVertexLess comparator;
    TVertexIterator vertexIterator(vertex.Vertices.begin());
    TVertexIterator vertexEnd(vertex.Vertices.end());
    TVector<std::tuple<TVertexIterator, TVertexIterator, TTagIterator>> state{{vertexIterator, vertexEnd, tagIterator}};
    while (!state.empty()) {
        std::tie(vertexIterator, vertexEnd, tagIterator) = state.back();
        state.pop_back();
        while (tagIterator != tagEnd && vertexIterator != vertexEnd) {
            if (comparator(*tagIterator, *vertexIterator)) {
                ++tagIterator;
            } else if (comparator(*vertexIterator, *tagIterator)) {
                ++vertexIterator;
            } else {
                Y_ASSERT(*tagIterator == vertexIterator->GetName());
                vertexIterator->Visit(visitor);
                state.emplace_back(vertexIterator->Vertices.begin(), vertexIterator->Vertices.end(), tagIterator);
                ++tagIterator;
                ++vertexIterator;
            }
        }
    }
}

void TTagTree::Clear() {
    Root.Clear();
}

void TTagTree::Out(IOutputStream& stream) const {
    Root.Out(stream);
}

template <>
void Out<TTagTree>(IOutputStream& stream,
                   TTypeTraits<TTagTree>::TFuncParam tree) {
    tree.Out(stream);
}
