#pragma once

#include <infra/yasm/common/labels/tags/interned_tag_name_set.h>
#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/tags/request_key.h>

#include <util/generic/set.h>
#include <util/generic/maybe.h>

namespace NZoom::NAggregators {
    class TTagTreeVisitor {
    public:
        virtual ~TTagTreeVisitor() = default;

        virtual void OnPath(NTags::TInternedTagNameSet tagNameSet, size_t position) = 0;
    };

    class TTagVertex;
    class TTagTree;

    struct TTagVertexLess {
        using is_transparent = void;

        bool operator()(const TTagVertex& lhs, const TTagVertex& rhs) const noexcept;
        bool operator()(const TTagVertex& lhs, const NTags::TTagName& rhs) const noexcept;
        bool operator()(const NTags::TTagName& lhs, const TTagVertex& rhs) const noexcept;
    };

    using TTagVertexSet = TSet<TTagVertex, TTagVertexLess>;

    class TTagVertex {
    public:
        friend TTagTree;

        TTagVertex(NTags::TTagName tagName, const TVector<NTags::TTagName>& tagNames)
            : Name(tagName)
            , Path(tagNames)
        {
        }

        NTags::TTagName GetName() const noexcept {
            return Name;
        }

        NTags::TInternedTagNameSet GetPath() const noexcept {
            return Path;
        }

        bool IsLeaf() const noexcept {
            return Position.Defined();
        }

        void MarkAsLeaf() const noexcept {
            if (!Position.Defined()) {
                Position.ConstructInPlace();
            }
        }

        void SetPosition(size_t position) const noexcept {
            Y_VERIFY(Position.Empty());
            Position = position;
        }

        size_t GetPosition() const noexcept {
            return *Position;
        }

        void Visit(TTagTreeVisitor& visitor) const;

        void Clear() const;

        void Out(IOutputStream& stream) const;

    private:
        NTags::TTagName Name;
        NTags::TInternedTagNameSet Path;
        mutable TMaybe<size_t, NMaybe::TPolicyUndefinedFail> Position;
        mutable TTagVertexSet Vertices;
    };

    class TTagTree {
    public:
        TTagTree();

        const TTagVertex& Create(const NTags::TRequestKey& requestKey);
        void Find(NTags::TInstanceKey instanceKey, TTagTreeVisitor& visitor) const;
        void Clear();

        void Out(IOutputStream& stream) const;

    private:
        using TTagIterator = TVector<NTags::TTagName>::const_iterator;
        using TVertexIterator = TTagVertexSet::const_iterator;

        void StepInto(const TTagVertex& vertex, TTagIterator tagIterator, const TTagIterator tagEnd, TTagTreeVisitor& visitor) const;

        TTagVertex Root;
    };
}
