#include "tag_tree.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NAggregators;
using namespace NTags;

namespace {
    class TRememberingVisitor : public TTagTreeVisitor {
    public:
        void OnPath(TInternedTagNameSet tagNameSet, size_t position) override {
            TagNames.emplace_back(tagNameSet, position);
        }

        bool Has(TInternedTagNameSet tagNameSet, size_t position) const {
            return Find(TagNames, std::make_pair(tagNameSet, position)) != TagNames.end();
        }

        size_t Size() const {
            return TagNames.size();
        }

        void Clear() {
            TagNames.clear();
        }

    private:
        TVector<std::pair<TInternedTagNameSet, size_t>> TagNames;
    };
}

Y_UNIT_TEST_SUITE(TTagTreeTest) {

    Y_UNIT_TEST(TestRepr) {
        {
            TTagTree tree;
            tree.Create(TRequestKey::FromString("itype=other")).SetPosition(0);
            UNIT_ASSERT_STRINGS_EQUAL(ToString(tree), "(itype 0 ())");
        }
        {
            TTagTree tree;
            tree.Create(TRequestKey::FromString("itype=other;a_tag=other")).SetPosition(0);
            UNIT_ASSERT_STRINGS_EQUAL(ToString(tree), "(itype null ((a_tag 0 ())))");
        }
        {
            TTagTree tree;
            tree.Create(TRequestKey::FromString("itype=other;b_tag=other;c_tag=other")).SetPosition(2);
            tree.Create(TRequestKey::FromString("itype=other;c_tag=other")).SetPosition(1);
            tree.Create(TRequestKey::FromString("itype=other;b_tag=other")).SetPosition(0);
            UNIT_ASSERT_STRINGS_EQUAL(ToString(tree), "(itype null ((b_tag 0 ((c_tag 2 ()))) (c_tag 1 ())))");
        }
        {
            TTagTree tree;
            tree.Create(TRequestKey::FromString("itype=other;b_tag=other;c_tag=other")).SetPosition(0);
            UNIT_ASSERT_STRINGS_EQUAL(ToString(tree), "(itype null ((b_tag null ((c_tag 0 ())))))");
        }
    }

    Y_UNIT_TEST(TestRootLookup) {
        TRememberingVisitor visitor;
        TTagTree tree;
        tree.Create(TRequestKey::FromString("itype=other")).SetPosition(0);
        tree.Find(TInstanceKey::FromNamed("something|prj=something"), visitor);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Size(), 1);
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{}), 0));
        visitor.Clear();
    }

    Y_UNIT_TEST(TestOneLevelLookup) {
        TRememberingVisitor visitor;
        TTagTree tree;
        tree.Create(TRequestKey::FromString("itype=other;prj=other")).SetPosition(0);
        tree.Create(TRequestKey::FromString("itype=other;ctype=other")).SetPosition(1);

        tree.Find(TInstanceKey::FromNamed("something|prj=something;ctype=something;geo=something"), visitor);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Size(), 2);
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{TStringBuf("prj")}), 0));
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{TStringBuf("ctype")}), 1));
        visitor.Clear();

        tree.Find(TInstanceKey::FromNamed("something|prj=something"), visitor);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Size(), 1);
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{TStringBuf("prj")}), 0));
        visitor.Clear();

        tree.Find(TInstanceKey::FromNamed("something|geo=something"), visitor);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Size(), 0);
        visitor.Clear();
    }

    Y_UNIT_TEST(TestMultiLevelLookup) {
        TRememberingVisitor visitor;
        TTagTree tree;
        tree.Create(TRequestKey::FromString("itype=other;prj=other;ctype=other;geo=other")).SetPosition(0);
        tree.Create(TRequestKey::FromString("itype=other;prj=other;ctype=other;tier=something")).SetPosition(1);
        tree.Create(TRequestKey::FromString("itype=other;geo=other")).SetPosition(2);

        tree.Find(TInstanceKey::FromNamed("something|prj=something;ctype=something;geo=something;tier=something"), visitor);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Size(), 3);
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{TStringBuf("prj"), TStringBuf("ctype"), TStringBuf("geo")}), 0));
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{TStringBuf("prj"), TStringBuf("ctype"), TStringBuf("tier")}), 1));
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{TStringBuf("geo")}), 2));
        visitor.Clear();

        tree.Find(TInstanceKey::FromNamed("something|prj=something;ctype=something;tier=something"), visitor);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Size(), 1);
        UNIT_ASSERT(visitor.Has(TInternedTagNameSet(TVector<TStringBuf>{TStringBuf("prj"), TStringBuf("ctype"), TStringBuf("tier")}), 1));
        visitor.Clear();

        tree.Find(TInstanceKey::FromNamed("something|prj=something;ctype=something;nanny=something"), visitor);
        UNIT_ASSERT_VALUES_EQUAL(visitor.Size(), 0);
        visitor.Clear();
    }

}
