#include "bit_stream.h"

namespace NYasmServer {
    void AddToBitStream(ui64 value, size_t bitsInValue, TString& stream, ui32& bitPosition) {
        size_t bitsAvailable = (bitPosition & 0b111) ? (8 - (bitPosition & 0b111)) : 0;
        bitPosition += bitsInValue;

        if (bitsInValue <= bitsAvailable) {
            // Everything fits inside the last byte
            stream[stream.size() - 1] = stream[stream.size() - 1] | (value << (bitsAvailable - bitsInValue));
            return;
        }

        size_t bitsLeft = bitsInValue;
        if (bitsAvailable > 0) {
            // Fill up the last byte
            stream[stream.size() - 1] = stream[stream.size() - 1] | (value >> (bitsInValue - bitsAvailable));
            bitsLeft -= bitsAvailable;
        }

        while (bitsLeft >= 8) {
            // Enough bits for a dedicated byte
            char ch = (value >> (bitsLeft - 8)) & 0xFF;
            stream.append(ch);
            bitsLeft -= 8;
        }

        if (bitsLeft != 0) {
            // Start a new byte with the rest of the bits
            char ch = (value & ((1 << bitsLeft) - 1)) << (8 - bitsLeft);
            stream.append(ch);
        }
    }

    ui64 ReadFromBitStreamLarge(const TString& stream, size_t& bitPosition, size_t bitsToRead) {
        Y_ASSERT(bitsToRead >= 8);

        size_t bitsAvailable = 8 - (bitPosition & 0b111);
        ui64 value = 0;

        if (bitsAvailable > 0) {
            value = stream.data()[bitPosition / 8] & ((1 << bitsAvailable) - 1);
            bitsToRead -= bitsAvailable;
            bitPosition += bitsAvailable;
        }

        while (bitsToRead >= 8) {
            value <<= 8;
            value |= stream.data()[bitPosition / 8] & 0xFF;
            bitsToRead -= 8;
            bitPosition += 8;
        }

        if (bitsToRead != 0) {
            value <<= bitsToRead;
            value |= (stream.data()[bitPosition / 8] >> (8 - bitsToRead)) & ((1 << bitsToRead) - 1);
            bitPosition += bitsToRead;
        }

        return value;
    }

    template <bool aligned>
    inline void WriteCompressedUint64Core(ui64 value, TString& stream, ui32& bitPosition) {
        do {
            ui8 byte = value & 0b01111111;
            value >>= 7;
            if (value > 0) {
                // set the first bit to 1 since there's some data left
                byte |= 0b10000000;
            }
            if (aligned) {
                AlignedAddToBitStreamUncompressed<ui8>(byte, stream, bitPosition);
            } else {
                AddToBitStreamUncompressed<ui8>(byte, stream, bitPosition);
            }
        } while (value);
    }

    void WriteCompressedUint64(ui64 value, TString& stream, ui32& bitPosition) {
        WriteCompressedUint64Core<false>(value, stream, bitPosition);
    }

    void AlignedWriteCompressedUint64(ui64 value, TString& stream, ui32& bitPosition) {
        WriteCompressedUint64Core<true>(value, stream, bitPosition);
    }

    template <bool aligned>
    inline ui64 ReadCompressedUint64Core(const TString& stream, size_t& bitPosition) {
        ui64 value = 0;
        bool dataLeft = true;
        size_t shift = 0;
        while (dataLeft) {
            ui64 byte;
            if (aligned) {
                byte = AlignedReadFromBitStreamUncompressed<ui8>(stream, bitPosition);
            } else {
                byte = ReadFromBitStreamUncompressed<ui8>(stream, bitPosition);
            }

            value |= (byte & 0b01111111) << shift;
            dataLeft = byte & 0b10000000;
            shift += 7;
        }
        return value;
    }

    ui64 ReadCompressedUint64(const TString& stream, size_t& bitPosition) {
        return ReadCompressedUint64Core<false>(stream, bitPosition);
    }

    ui64 AlignedReadCompressedUint64(const TString& stream, size_t& bitPosition) {
        return ReadCompressedUint64Core<true>(stream, bitPosition);
    }

    void TDeltaIntEncoder::Write(ui64 value, TString& stream, ui32& bitPosition) {
        i64 delta = (i64)value - (i64)PreviousValue;
        if (delta == 0) {
            // '0' bit - value hasn't changed
            AddToBitStream(0, 1, stream, bitPosition);
            return;
        }
        // '1' bit - value changed
        AddToBitStream(1, 1, stream, bitPosition);

        if (delta > 0) {
            // There are no zeros. Shift by one to fit in x number of bits
            delta--;
        }

        i64 absDelta = std::abs(delta);
        if (absDelta < 64) {
            AddToBitStream(0, 1, stream, bitPosition);
            AddToBitStream(delta + 64, 7, stream, bitPosition);
        } else {
            AddToBitStream(1, 1, stream, bitPosition);
            // now store the whole delta
            AddToBitStream(delta, 64, stream, bitPosition);
        }

        PreviousValue = value;
    }

    ui64 TDeltaIntDecoder::Read(const TString& stream, size_t& bitPosition) {
        ui64 notZeroDiff = ReadFromBitStream(stream, bitPosition, 1);
        if (!notZeroDiff) {
            // '0' header, value hasn't changed
            return PreviousValue;
        }
        ui64 fullValue = ReadFromBitStream(stream, bitPosition, 1);
        i64 delta;
        if (fullValue) {
            delta = ReadFromBitStream(stream, bitPosition, 64);
        } else {
            delta = ReadFromBitStream(stream, bitPosition, 7) - 64;
        }

        if (delta >= 0) {
            // [-128,127] becomes [-128,128] without the zero in the middle
            delta++;
        }

        PreviousValue += delta;
        return PreviousValue;
    }

    static const size_t LEADING_ZEROS_LENGTH_BITS = 5;
    static const size_t BLOCK_SIZE_LENGTH_BITS = 6;
    static const size_t MAX_LEADING_ZEROS_LENGTH_BITS = (1 << LEADING_ZEROS_LENGTH_BITS) - 1;
    static const size_t BLOCK_SIZE_ADJUSTMENT = 1;

    void TXorDoubleEncoder::Write(double value, TString& stream, ui32& bitPosition) {
        ui64* p = (ui64*)&value;
        ui64 xorWithPrevious = PreviousValue ^ *p;

        if (xorWithPrevious == 0) {
            AddToBitStream(0, 1, stream, bitPosition);
            return;
        }

        // '1', different value
        AddToBitStream(1, 1, stream, bitPosition);

        size_t leadingZeros = __builtin_clzll(xorWithPrevious);
        size_t trailingZeros = __builtin_ctzll(xorWithPrevious);

        if (leadingZeros > MAX_LEADING_ZEROS_LENGTH_BITS) {
            leadingZeros = MAX_LEADING_ZEROS_LENGTH_BITS;
        }

        size_t blockSize = 64 - leadingZeros - trailingZeros;
        size_t expectedSize = LEADING_ZEROS_LENGTH_BITS + BLOCK_SIZE_LENGTH_BITS + blockSize;
        size_t previousBlockInformationSize = 64 - PreviousValueLeadingZeros - PreviousValueTrailingZeros;
        if (leadingZeros >= PreviousValueLeadingZeros &&
            trailingZeros >= PreviousValueTrailingZeros &&
            previousBlockInformationSize < expectedSize) {
            // 0 since we use previous block information, '10' total
            AddToBitStream(0, 1, stream, bitPosition);

            ui64 blockValue = xorWithPrevious >> PreviousValueTrailingZeros;
            AddToBitStream(blockValue, previousBlockInformationSize, stream, bitPosition);

        } else {
            // 1 since we store all new sizes, '11' total
            AddToBitStream(1, 1, stream, bitPosition);
            // 5 bits for number of leading zeros
            AddToBitStream(leadingZeros, LEADING_ZEROS_LENGTH_BITS, stream, bitPosition);
            // 6 bits for block size, adjust to fit in 6 bits. There will never be a zero size block
            AddToBitStream(blockSize - BLOCK_SIZE_ADJUSTMENT, BLOCK_SIZE_LENGTH_BITS, stream, bitPosition);

            // actual value without trailing zeros
            ui64 blockValue = xorWithPrevious >> trailingZeros;
            AddToBitStream(blockValue, blockSize, stream, bitPosition);

            PreviousValueTrailingZeros = trailingZeros;
            PreviousValueLeadingZeros = leadingZeros;
        }

        PreviousValue = *p;
    }

    double TXorDoubleDecoder::Read(const TString& stream, size_t& bitPosition) {
        ui64 notZeroDiff = ReadFromBitStream(stream, bitPosition, 1);
        if (!notZeroDiff) {
            // '0' header, value hasn't changed
            return *(double*)&PreviousValue;
        }

        ui64 containsBlockSizes = ReadFromBitStream(stream, bitPosition, 1);
        ui64 xorValue;
        if (!containsBlockSizes) {
            // '10', uses block sizes from previous block
            xorValue = ReadFromBitStream(stream, bitPosition, 64 - PreviousLeadingZeros - PreviousTrailingZeros);
            xorValue <<= PreviousTrailingZeros;
        } else {
            // '11', contains block sizes
            ui64 leadingZeros = ReadFromBitStream(stream, bitPosition, LEADING_ZEROS_LENGTH_BITS);
            ui64 blockSize = ReadFromBitStream(stream, bitPosition, BLOCK_SIZE_LENGTH_BITS) + BLOCK_SIZE_ADJUSTMENT;
            PreviousTrailingZeros = 64 - blockSize - leadingZeros;
            xorValue = ReadFromBitStream(stream, bitPosition, blockSize);
            xorValue <<= PreviousTrailingZeros;
            PreviousLeadingZeros = leadingZeros;
        }

        ui64 value = xorValue ^ PreviousValue;
        PreviousValue = value;

        return *(double*)&value;
    }
} // namespace NYasmServer
