#pragma once

#include <util/generic/maybe.h>
#include <util/generic/cast.h>
#include <util/system/byteorder.h>
#include <util/string/builder.h>

namespace NYasmServer {
    namespace {
        template <class T>
        inline ui64 ReinterpetUint(std::enable_if_t<!std::is_integral<T>::value, T> value) {
            return BitCast<ui64, T>(value);
        }

        template <class T>
        inline ui64 ReinterpetUint(std::enable_if_t<std::is_integral<T>::value, T> value) {
            return (ui64)value;
        }
    } // namespace

    void AddToBitStream(ui64 value, size_t bitsInValue, TString& stream, ui32& bitPosition);

    template <class T>
    inline void AddToBitStreamUncompressed(T value, TString& stream, ui32& bitPosition) {
        ui64 converted = ReinterpetUint<T>(value);
        if (sizeof(T) < 8) {
            converted &= (1ul << (sizeof(T) * 8)) - 1;
        }
        AddToBitStream(converted, sizeof(T) * 8, stream, bitPosition);
    }

    template <class T>
    inline void AlignedAddToBitStreamUncompressed(T value, TString& stream, ui32& bitPosition) {
        Y_ASSERT((bitPosition & 0b111) == 0);
        ui64 converted = ReinterpetUint<T>(value);
        converted = SwapBytes(converted);
        converted >>= (8 - sizeof(T)) * 8;

        stream.append((const char*)&converted, sizeof(T));
        bitPosition += 8 * sizeof(T);
    }

    inline ui32 AlignToByteBoundary(ui32 bitPosition) {
        return (bitPosition + 7) & (~0b111);
    }

    inline ui64 ReadFromBitStreamSmall(const TString& stream, size_t& bitPosition, size_t bitsToRead) {
        ui64 value = 0;
        for (size_t i = 0; i < bitsToRead; i++) {
            value <<= 1;
            size_t bit = (stream.data()[bitPosition / 8] >> (0b111 - (bitPosition & 0b111))) & 1;
            value += bit;
            bitPosition++;
        }
        return value;
    }

    ui64 ReadFromBitStreamLarge(const TString& stream, size_t& bitPosition, size_t bitsToRead);

    inline ui64 ReadFromBitStream(const TString& stream, size_t& bitPosition, size_t bitsToRead) {
        if ((bitPosition + bitsToRead) > stream.size() * 8) {
            ythrow yexception() << "Trying to read past buffer";
        }
        // Y_VERIFY((bitPosition + bitsToRead) <= stream.Size() * 8);

        if (bitsToRead >= 8) {
            return ReadFromBitStreamLarge(stream, bitPosition, bitsToRead);
        }
        return ReadFromBitStreamSmall(stream, bitPosition, bitsToRead);
    }

    inline static TString BitStreamToString(const TString& stream, size_t maxLength = 0) {
        // not very efficient but it's a debug function anyway
        TStringBuilder out;
        out.reserve(stream.size() * 8);
        size_t position = 0;
        while (position < (stream.size() * 8) && (maxLength == 0 || position < maxLength)) {
            out << (ReadFromBitStream(stream, position, 1) ? '1' : '0');
        }
        return out;
    }

    template <class T>
    inline T ReadFromBitStreamUncompressed(const TString& stream, size_t& bitPosition) {
        ui64 value = ReadFromBitStream(stream, bitPosition, sizeof(T) * 8);
        return *(T*)&value;
    }

    template <class T>
    inline T AlignedReadFromBitStreamUncompressed(const TString& stream, size_t& bitPosition) {
        Y_ASSERT((bitPosition & 0b111) == 0);
        ui64 value = 0;
        memcpy(&value, stream.data() + bitPosition / 8, sizeof(T));
        value = SwapBytes(value);
        value >>= (8 - sizeof(T)) * 8;
        bitPosition += sizeof(T) * 8;
        return *(T*)&value;
    }

    void WriteCompressedUint64(ui64 value, TString& stream, ui32& bitPosition);

    void AlignedWriteCompressedUint64(ui64 value, TString& stream, ui32& bitPosition);

    ui64 ReadCompressedUint64(const TString& stream, size_t& bitPosition);

    ui64 AlignedReadCompressedUint64(const TString& stream, size_t& bitPosition);

    class TDeltaIntEncoder {
    public:
        using TValueType = ui64;

        TDeltaIntEncoder() {
            Clear();
        }

        void Write(ui64 value, TString& stream, ui32& bitPosition);

        void Clear() {
            PreviousValue = 0;
        }

    private:
        ui64 PreviousValue;
    };

    class TDeltaIntDecoder {
    public:
        using TValueType = ui64;

        ui64 Read(const TString& stream, size_t& bitPosition);

        void Clear() {
            PreviousValue = 0;
        }

    private:
        ui64 PreviousValue = 0;
    };

    class TXorDoubleEncoder {
    public:
        using TValueType = double;

        void Write(double value, TString& stream, ui32& bitPosition);

        void Clear() {
            PreviousValue = 0;
            PreviousValueLeadingZeros = sizeof(double) * 8;
            PreviousValueTrailingZeros = 0;
        }

    private:
        ui64 PreviousValue = 0;
        ui8 PreviousValueLeadingZeros = sizeof(double) * 8;
        ui8 PreviousValueTrailingZeros = 0;
    };

    class TXorDoubleDecoder {
    public:
        using TValueType = double;

        double Read(const TString& stream, size_t& bitPosition);

        void Clear() {
            PreviousValue = 0;
            PreviousLeadingZeros = sizeof(double) * 8;
            PreviousTrailingZeros = 0;
        }

    private:
        ui64 PreviousValue = 0;
        size_t PreviousLeadingZeros = sizeof(double) * 8;
        size_t PreviousTrailingZeros = 0;
    };
} // namespace NYasmServer
