#include "bit_stream.h"

#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NYasmServer {

    Y_UNIT_TEST_SUITE(TestSimpleBitStream) {
        Y_UNIT_TEST(TestReadingAndWritingSomeBits) {
            TString stream;
            ui32 writePosition = 0;

            AddToBitStream(1, 1, stream, writePosition);
            AddToBitStream(0b101, 3, stream, writePosition);
            AddToBitStream(0b0, 1, stream, writePosition);
            double value = 123.432;
            AddToBitStream(*(ui64*)&value, 64, stream, writePosition);

            size_t readPosition = 0;
            UNIT_ASSERT_EQUAL(ReadFromBitStream(stream, readPosition, 1), 1);
            UNIT_ASSERT_EQUAL(ReadFromBitStream(stream, readPosition, 3), 0b101);
            UNIT_ASSERT_EQUAL(ReadFromBitStream(stream, readPosition, 1), 0);
            ui64 result = ReadFromBitStream(stream, readPosition, 64);
            UNIT_ASSERT_DOUBLES_EQUAL(*(double*)&result, 123.432, 0.001);
        }

        Y_UNIT_TEST(TestSmallNegativesNotAffectingOtherBytes) {
            TString stream;
            ui32 writePosition = 0;
            // some padding
            AddToBitStream(1, 6, stream, writePosition);
            // actual value
            AddToBitStream(15, 8, stream, writePosition);
            AddToBitStreamUncompressed<i16>(-5, stream, writePosition);

            size_t readPosition = 0;
            UNIT_ASSERT_EQUAL(ReadFromBitStream(stream, readPosition, 6), 1);
            // this will fail if writing -5 there affected the already written stream somehow
            UNIT_ASSERT_EQUAL(ReadFromBitStream(stream, readPosition, 8), 15);

            UNIT_ASSERT_EQUAL(ReadFromBitStreamUncompressed<i16>(stream, readPosition), -5);
        }

        Y_UNIT_TEST(TestAligningToByteBoundary) {
            UNIT_ASSERT_EQUAL(AlignToByteBoundary(0), 0);
            UNIT_ASSERT_EQUAL(AlignToByteBoundary(1), 8);
            UNIT_ASSERT_EQUAL(AlignToByteBoundary(7), 8);
            UNIT_ASSERT_EQUAL(AlignToByteBoundary(8), 8);
            UNIT_ASSERT_EQUAL(AlignToByteBoundary(9), 16);
        }
    }

    Y_UNIT_TEST_SUITE(TestVarintCompression) {
        Y_UNIT_TEST(TestReadingAndWritingSimpleIntegers) {
            TString stream;
            ui32 writePosition = 0;

            // 1 byte
            WriteCompressedUint64(0, stream, writePosition);
            UNIT_ASSERT_EQUAL(stream.Size(), 1);

            // 1 byte
            WriteCompressedUint64(5, stream, writePosition);
            UNIT_ASSERT_EQUAL(stream.Size(), 2);

            // 2 bytes
            WriteCompressedUint64(275, stream, writePosition);
            UNIT_ASSERT_EQUAL(stream.Size(), 4);

            // 4 bytes
            WriteCompressedUint64(12936123, stream, writePosition);
            UNIT_ASSERT_EQUAL(stream.Size(), 8);

            size_t readPosition = 0;
            UNIT_ASSERT_EQUAL(ReadCompressedUint64(stream, readPosition), 0);
            UNIT_ASSERT_EQUAL(ReadCompressedUint64(stream, readPosition), 5);
            UNIT_ASSERT_EQUAL(ReadCompressedUint64(stream, readPosition), 275);
            UNIT_ASSERT_EQUAL(ReadCompressedUint64(stream, readPosition), 12936123);
        }
    }

    Y_UNIT_TEST_SUITE(TestDoubleXorCompression) {
        Y_UNIT_TEST(TestStreamOfIdenticalValues) {
            TXorDoubleEncoder encoder;
            TString stream;
            ui32 writePosition = 0;
            for (size_t i = 0; i < 1000; i++) {
                encoder.Write(15.5, stream, writePosition);
            }

            TXorDoubleDecoder decoder;
            size_t readPosition = 0;
            for (size_t i = 0; i < 1000; i++) {
                UNIT_ASSERT_DOUBLES_EQUAL(decoder.Read(stream, readPosition), 15.5, 0.001);
            }
        }

        Y_UNIT_TEST(TestStreamOfIncrementingValues) {
            TXorDoubleEncoder encoder;
            TString stream;
            ui32 writePosition = 0;
            for (size_t i = 0; i < 1000; i++) {
                encoder.Write((double)i + 0.05, stream, writePosition);
            }

            TXorDoubleDecoder decoder;
            size_t readPosition = 0;
            for (size_t i = 0; i < 1000; i++) {
                UNIT_ASSERT_DOUBLES_EQUAL(decoder.Read(stream, readPosition), (double)i + 0.05, 0.001);
            }
        }
    }
} // namespace NYasmServer
