#include "chunk.h"
#include "zoom_converters.h"

#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>

#include <util/generic/xrange.h>

#include <cmath>

namespace NYasmServer {
    void THistogramChunk::OnStoreSmall(const TVector<double>& values, size_t zeros) {
        NYasmServer::TSimpleHistogram hist;
        hist.SetZeroCount(zeros);
        hist.MutableValues() = values;
        Writer.PushValue(Timestamp, std::move(hist));
    }

    void THistogramChunk::OnStoreNormal(const TVector<double>& values, size_t zeros, i16 startPower) {
        NYasmServer::TLogHistogram hist;
        hist.SetZeroCount(zeros);
        hist.SetStartPower(startPower);
        hist.MutableWeights().resize(values.size());
        for (size_t i = 0; i < values.size(); i++) {
            hist.MutableWeights()[i] = (ui64)std::round(values[i]);
        }
        Writer.PushValue(Timestamp, std::move(hist));
    }

    void THistogramChunk::OnStoreUgram(const NZoom::NHgram::TUgramBuckets& buckets) {
        const NZoom::NHgram::TUgramBuckets& compressedBuckets = NZoom::NHgram::TUgramCompressor::GetInstance().Compress(buckets);

        NYasmServer::TUserHistogram hist;
        if (!compressedBuckets.empty()) {
            hist.MutableBuckets().reserve(compressedBuckets.size());

            for (size_t i = 0; i < compressedBuckets.size() - 1; i++) {
                auto& current = compressedBuckets[i];
                hist.AppendBucket(current.LowerBound, (ui64)std::round(current.Weight));

                if (current.UpperBound != compressedBuckets[i + 1].LowerBound) {
                    hist.AppendBucket(current.UpperBound, 0);
                }
            }

            // last bucket
            auto& last = compressedBuckets.back();
            hist.AppendBucket(last.LowerBound, (ui64)std::round(last.Weight));
            if (last.LowerBound != last.UpperBound) {
                hist.AppendBucket(last.UpperBound, 0);
            }
        }
        Writer.PushValue(Timestamp, std::move(hist));
    }

    TVector<NZoom::NValue::TValue> DecodeChunk(const TString& data, ESeriesKind seriesKind, size_t valuesCount) {
        TVector<NZoom::NValue::TValue> values(Reserve(valuesCount));
        DecodeChunkAndVisit(data, seriesKind, valuesCount, [&values](auto&& value) {
            values.emplace_back(std::forward<decltype(value)>(value));
        });
        return values;
    }
}
