#pragma once

#include <infra/yasm/common/points/value/abstract/value.h>

#include <infra/yasm/zoom/components/compression/series.h>
#include <infra/yasm/zoom/components/compression/bit_stream.h>
#include <infra/yasm/zoom/components/compression/histograms.h>
#include <infra/yasm/zoom/components/compression/counted_sum.h>
#include <infra/yasm/zoom/components/compression/zoom_converters.h>

#include <util/datetime/base.h>
#include <util/generic/ptr.h>

namespace NYasmServer {
    class TBaseChunk : public NZoom::NValue::IUpdatable {
    public:
        virtual ~TBaseChunk() = default;

        virtual void SetTimestamp(ui64 timestamp) = 0;
        virtual ui64 GetStartTime() const = 0;
        virtual const TString& GetData() const = 0;
        virtual NYasmServer::ESeriesKind GetType() const = 0;
        virtual ui32 GetValuesWritten() const = 0;

        void MulFloat(const double) override {
            MulNone();
        }
        void MulVec(const TVector<double>&) override {
            MulNone();
        }
        void MulCountedSum(const double, const ui64) override {
            MulNone();
        }
        void MulHgram(const NZoom::NHgram::THgram&) override {
            MulNone();
        }
    };

    template <class TEncoder>
    class TWriter {
    public:
        using TValueType = typename TEncoder::TValueType;

        TWriter(ui64 startTime)
            : StartTime(startTime)
            , ValuesWritten(0)
            , BitsWritten(0)
        {
        }

        void PushValue(ui64 timestamp, TValueType&& value) {
            WriteNullsUntil(timestamp);
            // '1' - not a null value
            NYasmServer::AddToBitStream(1, 1, Buffer, BitsWritten);
            Encoder.Write(std::move(value), Buffer, BitsWritten);
            ValuesWritten++;
        }

        void PushNull(ui64 timestamp) {
            WriteNullsUntil(timestamp);
            AppendNull();
        }

        void AppendNull() {
            // append a single 0 bit
            NYasmServer::AddToBitStream(0, 1, Buffer, BitsWritten);
            ValuesWritten++;
        }

        const TString& GetData() const {
            return Buffer;
        }

        ui64 GetStartTime() const {
            return StartTime;
        }

        ui32 GetValuesWritten() const {
            return ValuesWritten;
        }

    private:
        void WriteNullsUntil(ui64 startTime) {
            // push nulls for every missing value
            const size_t valuesToWrite = startTime - StartTime - ValuesWritten;
            for (size_t i = 0; i < valuesToWrite; i++) {
                AppendNull();
            }
        }

        TString Buffer;
        ui64 StartTime;
        ui32 ValuesWritten;
        // number of bits we have written into the current buffer
        ui32 BitsWritten;
        // value encoder
        TEncoder Encoder;
    };

    class TDoubleChunk final : public TBaseChunk {
    public:
        TDoubleChunk(ui64 startTime)
            : Writer(startTime)
        {
        }

        void MulNone() override {
            Writer.PushNull(Timestamp);
        }

        void MulFloat(const double value) override {
            Writer.PushValue(Timestamp, (double)value);
        }

        void SetTimestamp(ui64 timestamp) override {
            Timestamp = timestamp;
        }

        ui64 GetStartTime() const override {
            return Writer.GetStartTime();
        }

        const TString& GetData() const override {
            return Writer.GetData();
        }

        NYasmServer::ESeriesKind GetType() const override {
            return NYasmServer::ESeriesKind::Double;
        }

        ui32 GetValuesWritten() const override {
            return Writer.GetValuesWritten();
        }

    private:
        TWriter<NYasmServer::TXorDoubleEncoder> Writer;
        ui64 Timestamp = 0;
    };

    class TCountedSumChunk final : public TBaseChunk {
    public:
        TCountedSumChunk(ui64 startTime)
            : Writer(startTime)
        {
        }

        void MulNone() override {
            Writer.PushNull(Timestamp);
        }

        void MulCountedSum(const double sum, const ui64 count) override {
            Writer.PushValue(Timestamp, NYasmServer::TCountedSum(count, sum));
        }

        void SetTimestamp(ui64 timestamp) override {
            Timestamp = timestamp;
        }

        ui64 GetStartTime() const override {
            return Writer.GetStartTime();
        }

        const TString& GetData() const override {
            return Writer.GetData();
        }

        NYasmServer::ESeriesKind GetType() const override {
            return NYasmServer::ESeriesKind::CountedSum;
        }

        ui32 GetValuesWritten() const override {
            return Writer.GetValuesWritten();
        }

    private:
        TWriter<NYasmServer::TCountedSumEncoder> Writer;
        ui64 Timestamp = 0;
    };

    class THistogramChunk final : public TBaseChunk, public NZoom::NHgram::IHgramStorageCallback {
    public:
        THistogramChunk(ui64 startTime)
            : Writer(startTime)
        {
        }

        void MulNone() override {
            Writer.PushNull(Timestamp);
        }

        void MulVec(const TVector<double>& values) override {
            NYasmServer::TSimpleHistogram hist;
            hist.MutableValues() = values;
            Writer.PushValue(Timestamp, std::move(hist));
        }

        void MulHgram(const NZoom::NHgram::THgram& value) override {
            value.Store(*this);
        }

        void OnStoreSmall(const TVector<double>& values, size_t zeros) override;
        void OnStoreNormal(const TVector<double>& values, size_t zeros, i16 startPower) override;
        void OnStoreUgram(const NZoom::NHgram::TUgramBuckets& buckets) override;

        void SetTimestamp(ui64 timestamp) override {
            Timestamp = timestamp;
        }

        ui64 GetStartTime() const override {
            return Writer.GetStartTime();
        }

        const TString& GetData() const override {
            return Writer.GetData();
        }

        NYasmServer::ESeriesKind GetType() const override {
            return NYasmServer::ESeriesKind::Histogram;
        }

        ui32 GetValuesWritten() const override {
            return Writer.GetValuesWritten();
        }

    private:
        TWriter<NYasmServer::THistogramEncoder> Writer;
        ui64 Timestamp = 0;
    };

    TVector<NZoom::NValue::TValue> DecodeChunk(const TString& data, ESeriesKind seriesKind, size_t valuesCount);

    template<typename TValueContainer>
    TString EncodeChunk(const TValueContainer& values, TMaybe<ESeriesKind>& seriesKind, bool allNoneToEmptyString) {
        TValueTyper typer;
        seriesKind.Clear();
        for (const auto& value : values) {
            value.Update(typer);
            if (typer.GetType().Defined()) {
                seriesKind = typer.GetType();
                break;
            }
        }
        if (!seriesKind.Defined() && allNoneToEmptyString) {
            return TString();
        }
        // if all the values are null we write them using double writer since nulls are written in the same way
        // by all writers
        const ESeriesKind encodeAsType = (seriesKind.Defined()) ? seriesKind.GetRef() : ESeriesKind::Double;
        ui64 valueIdx = 0;
        THolder<TBaseChunk> chunk;
        switch (encodeAsType) {
            case ESeriesKind::Histogram:
                chunk = MakeHolder<THistogramChunk>(valueIdx);
                break;
            case ESeriesKind::CountedSum:
                chunk = MakeHolder<TCountedSumChunk>(valueIdx);
                break;
            case ESeriesKind::Double:
                chunk = MakeHolder<TDoubleChunk>(valueIdx);
                break;
        }

        for (const auto& value: values) {
            value.Update(*chunk);

            ++valueIdx;
            chunk->SetTimestamp(valueIdx);
        }

        return chunk->GetData();
    }

    template<class TDecoder>
    class TChunkDecoder {
    public:
        template <typename TVisitor>
        static void DecodeAndVisit(const TString& data, size_t valuesCount, TVisitor visitor) {
            TDecoder decoder;
            size_t bitPosition = 0;
            for (const auto idx : xrange(valuesCount)) {
                bool notNull = ReadFromBitStream(data, bitPosition, 1) != 0;
                if (!notNull) {
                    visitor(NZoom::NValue::TValue());
                } else {
                    visitor(ToZoom(decoder.Read(data, bitPosition)));
                }
                Y_UNUSED(idx);
            }
        }
        static TVector<NZoom::NValue::TValue> Decode(const TString& data, size_t valuesCount) {
            TVector<NZoom::NValue::TValue> values(Reserve(valuesCount));
            DecodeAndVisit(data, valuesCount, [&values](auto&& value) {
                values.emplace_back(std::forward<decltype(value)>(value));
            });
            return values;
        }
    };

    template <typename TVisitor>
    void DecodeChunkAndVisit(const TString& data, ESeriesKind seriesKind, size_t valuesCount, TVisitor visitor) {
        switch (seriesKind) {
            case NYasmServer::ESeriesKind::Double: {
                TChunkDecoder<TXorDoubleDecoder>::DecodeAndVisit(data, valuesCount, visitor);
                break;
            }
            case NYasmServer::ESeriesKind::CountedSum: {
                TChunkDecoder<TCountedSumDecoder>::DecodeAndVisit(data, valuesCount, visitor);
                break;
            }
            case NYasmServer::ESeriesKind::Histogram: {
                TChunkDecoder<THistogramDecoder>::DecodeAndVisit(data, valuesCount, visitor);
                break;
            }
        }
    }
} // namespace NYasmServer
