#pragma once

#include "zoom_types.h"

#include <infra/yasm/common/points/hgram/hgram.h>
#include <infra/yasm/common/points/value/types.h>

namespace NYasmServer {
    using TValue = NZoom::NValue::TValue;

    inline static NZoom::NHgram::TUgramBuckets ConvertUgramBuckets(const TVector<TUserHistogram::TBucket>& buckets) {
        NZoom::NHgram::TUgramBuckets result;
        if (buckets.empty()) {
            return result;
        }
        // this probably allocates too much memory but we don't know the actual size in advance so...
        result.reserve(buckets.size());

        for (size_t i = 0; i < buckets.size() - 1; i++) {
            result.emplace_back(buckets[i].LowerBound, buckets[i + 1].LowerBound, buckets[i].Weight);
            if (buckets[i + 1].Weight == 0) {
                i++; // skip the next one, it was just used to store upper bound
            }
        }

        if (buckets.back().Weight != 0) {
            // last one also stores some value and not just the upper bound
            result.emplace_back(buckets.back().LowerBound, buckets.back().LowerBound, buckets.back().Weight);
        }

        return result;
    }

    template <class T>
    inline TValue ToZoom(const T&) {
        ythrow yexception() << "Not implemented";
    }

    template <>
    inline TValue ToZoom<double>(const double& value) {
        return TValue(value);
    }

    template <>
    inline TValue ToZoom<TCountedSum>(const TCountedSum& value) {
        return TValue(value.GetSum(), value.GetCount());
    }

    template <>
    inline TValue ToZoom<THistogram>(const THistogram& value) {
        switch (value.GetKind()) {
            case EHistogramKind::Log: {
                auto& asLog = value.AsLogHistogram();
                // we really should get rid of this conversion somehow to avoid this useless vector copy
                TVector<double> weights;
                weights.reserve(asLog.GetWeights().size());
                for (const auto& weight : asLog.GetWeights()) {
                    weights.push_back((double)weight);
                }
                return TValue(NZoom::NHgram::THgram::Normal(std::move(weights), asLog.GetZeroCount(), asLog.GetStartPower()));
            }
            case EHistogramKind::Simple: {
                auto& asSimple = value.AsSimpleHistogram();
                auto values = asSimple.GetValues();
                return TValue(NZoom::NHgram::THgram::Small(std::move(values), asSimple.GetZeroCount()));
            }
            case EHistogramKind::User: {
                return TValue(NZoom::NHgram::THgram::Ugram(ConvertUgramBuckets(value.AsUserHistogram().GetBuckets())));
            }
        }
    }
} // namespace NYasmServer
