#pragma once

#include <util/generic/variant.h>
#include <util/generic/vector.h>
#include <util/system/types.h>

namespace NYasmServer {

    class TCountedSum {
    public:
        TCountedSum(ui64 count = 0, double sum = 0)
            : Count(count)
            , Sum(sum) {
        }

        ui64 GetCount() const {
            return Count;
        }

        double GetSum() const {
            return Sum;
        }

        bool operator==(const TCountedSum& other) const {
            return Count == other.Count && Sum == other.Sum;
        }

    private:
        ui64 Count;
        double Sum;
    };

    enum class EHistogramKind {
        Simple,
        Log,
        User
    };

    class THistogramImpl {
    public:
        virtual EHistogramKind GetKind() const = 0;
        virtual ~THistogramImpl() = default;
    };

    class TSimpleHistogram : public THistogramImpl {
    public:
        enum class EKind {
            Empty = 0,
            ZeroCount = 1,
            SingleValue = 2,
            Normal = 3
        };

        TVector<double>& MutableValues() {
            return Values;
        }

        const TVector<double>& GetValues() const {
            return Values;
        }

        ui64 GetZeroCount() const {
            return ZeroCount;
        }

        void SetZeroCount(ui64 value) {
            ZeroCount = value;
        }

        virtual EHistogramKind GetKind() const override {
            return EHistogramKind::Simple;
        }

        bool IsEmpty() const {
            return GetSimpleKind() == EKind::Empty;
        }

        bool IsZeroCount() const {
            return GetSimpleKind() == EKind::ZeroCount;
        }

        bool IsSingleValue() const {
            return GetSimpleKind() == EKind::SingleValue;
        }

        EKind GetSimpleKind() const {
            if (Values.empty()) {
                return ZeroCount == 0 ? EKind::Empty : EKind::ZeroCount;
            } else if (ZeroCount == 0 && Values.size() == 1) {
                return EKind::SingleValue;
            }
            return EKind::Normal;
        }

        bool operator==(const TSimpleHistogram& other) const {
            return ZeroCount == other.ZeroCount && Values == other.Values;
        }

    private:
        TVector<double> Values;
        ui64 ZeroCount = 0;
    };

    class TLogHistogram : public THistogramImpl {
    public:
        TVector<ui64>& MutableWeights() {
            return Weights;
        }

        const TVector<ui64>& GetWeights() const {
            return Weights;
        }

        ui64 GetZeroCount() const {
            return ZeroCount;
        }

        void SetZeroCount(ui64 value) {
            ZeroCount = value;
        }

        i16 GetStartPower() const {
            return StartPower;
        }

        void SetStartPower(i16 value) {
            StartPower = value;
        }

        virtual EHistogramKind GetKind() const override {
            return EHistogramKind::Log;
        }

        bool operator==(const TLogHistogram& other) const {
            return ZeroCount == other.ZeroCount &&
                   StartPower == other.StartPower &&
                   Weights == other.Weights;
        }

    private:
        TVector<ui64> Weights;
        ui64 ZeroCount = 0;
        i16 StartPower = 0;
    };

    class TUserHistogram : public THistogramImpl {
    public:
        struct TBucket {
            double LowerBound = 0;
            ui64 Weight = 0;

            bool operator==(const TBucket& other) const {
                return LowerBound == other.LowerBound && Weight == other.Weight;
            }
        };

        TVector<TBucket>& MutableBuckets() {
            return Buckets;
        }

        const TVector<TBucket>& GetBuckets() const {
            return Buckets;
        }

        void AppendBucket(double lowerBound, ui64 weight) {
            Buckets.push_back(TBucket{.LowerBound = lowerBound, .Weight = weight});
        }

        virtual EHistogramKind GetKind() const override {
            return EHistogramKind::User;
        }

        bool operator==(const TUserHistogram& other) const {
            return Buckets == other.Buckets;
        }

    private:
        TVector<TBucket> Buckets;
    };

    class THistogram {
    public:
        THistogram()  = default;
        THistogram(THistogram&& other) = default;

        THistogram(TLogHistogram value)
            : Value(MakeHolder<TLogHistogram>(std::move(value))) {
        }

        THistogram(TUserHistogram value)
            : Value(MakeHolder<TUserHistogram>(std::move(value))) {
        }

        THistogram(TSimpleHistogram value)
            : Value(MakeHolder<TSimpleHistogram>(std::move(value))) {
        }

        THistogram(const THistogram& other) {
            if (other.IsNull()) {
                Value = nullptr;
                return;
            }
            switch (other.GetKind()) {
                case EHistogramKind::Simple:
                    Value = MakeHolder<TSimpleHistogram>(other.AsSimpleHistogram());
                    break;
                case EHistogramKind::User:
                    Value = MakeHolder<TUserHistogram>(other.AsUserHistogram());
                    break;
                case EHistogramKind::Log:
                    Value = MakeHolder<TLogHistogram>(other.AsLogHistogram());
                    break;
            }
        }

        THistogram& operator=(THistogram&& other) = default;

        const TLogHistogram& AsLogHistogram() const {
            Y_ASSERT(GetKind() == EHistogramKind::Log);
            return *static_cast<TLogHistogram*>(Value.Get());
        }

        TLogHistogram& MutableLogHistogram() {
            Y_ASSERT(GetKind() == EHistogramKind::Log);
            return *static_cast<TLogHistogram*>(Value.Get());
        }

        const TUserHistogram& AsUserHistogram() const {
            Y_ASSERT(GetKind() == EHistogramKind::User);
            return *static_cast<TUserHistogram*>(Value.Get());
        }

        TUserHistogram& MutableUserHistogram() {
            Y_ASSERT(GetKind() == EHistogramKind::User);
            return *static_cast<TUserHistogram*>(Value.Get());
        }

        const TSimpleHistogram& AsSimpleHistogram() const {
            Y_ASSERT(GetKind() == EHistogramKind::Simple);
            return *static_cast<TSimpleHistogram*>(Value.Get());
        }

        TSimpleHistogram& MutableSimpleHistogram() {
            Y_ASSERT(GetKind() == EHistogramKind::Simple);
            return *static_cast<TSimpleHistogram*>(Value.Get());
        }

        EHistogramKind GetKind() const {
            Y_VERIFY(!IsNull());
            return Value->GetKind();
        }

        bool IsNull() const {
            return Value == nullptr;
        }

        void Reset() {
            Value = nullptr;
        }

        bool operator==(const THistogram& other) const noexcept {
            if (IsNull() != other.IsNull()) {
                // one null, other isn't
                return false;
            }
            if (IsNull()) {
                // both are null
                return true;
            }
            // both are not null
            if (GetKind() != other.GetKind()) {
                return false;
            }
            switch (GetKind()) {
                case EHistogramKind::Simple:
                    return AsSimpleHistogram() == other.AsSimpleHistogram();
                case EHistogramKind::User:
                    return AsUserHistogram() == other.AsUserHistogram();
                case EHistogramKind::Log:
                    return AsLogHistogram() == other.AsLogHistogram();
            }
        }

    private:
        THolder<THistogramImpl> Value;
    };

} // namespace NYasmServer
