#include "agent.h"

using namespace NZoom::NContainers;
using namespace NZoom::NAccumulators;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace {

    static constexpr TDuration DEFAULT_SIGNAL_TTL_VALUE = TDuration::Seconds(30);

    template <typename TMap>
    void CleanMap(TMap& map, const TInstant now) {
        for (auto it = map.begin(); it != map.end();) {
            auto prev = it;
            ++it;
            if (prev->second.IsExpiredTTL(now)) {
                if (prev->second.IsExpired(now)) {
                    map.erase(prev);
                } else {
                    prev->second.Clean();
                }
            }
        }
    }

    inline void CalcTimes(const TDuration ttl, TInstant& ttlInstant, TInstant& expireTime) {
        const TInstant now = TInstant::Now();
        ttlInstant = now + ttl;
        const TInstant maxExpireTime = now + DEFAULT_SIGNAL_TTL_VALUE;
        expireTime = (ttlInstant > maxExpireTime) ? ttlInstant : maxExpireTime;
    }
}

TAgentWrappedAccumulator::TAgentWrappedAccumulator(NZoom::NAccumulators::EAccumulatorType type)
    : Accumulator(type)
{
}

void TAgentWrappedAccumulator::Mul(const TValueRef& value, const TInstant expireTime, const TInstant ttl) {
    Accumulator.Mul(value);
    ExpireTime = expireTime;
    TTL = ttl;
}

bool TAgentWrappedAccumulator::IsExpired(const TInstant now) const noexcept {
    return now > ExpireTime;
}

bool TAgentWrappedAccumulator::IsExpiredTTL(const TInstant now) const noexcept {
    return now > TTL;
}

void TAgentWrappedAccumulator::Clean() {
    Accumulator.Clean();
}

NZoom::NValue::TValueRef TAgentWrappedAccumulator::GetValue() const {
    return Accumulator.GetValue();
}

TAgentContainer::TAgentContainer(const NZoom::NYasmConf::TTypeConf& conf, const bool hostAggregation)
    : Storage(conf.GetSignals(),
        conf.GetPatterns(), (hostAggregation ? EAggregationMethod::Group : EAggregationMethod::Agent))
{
}

void TAgentContainer::Mul(const TAgentContainer& other) {
    MulTTL(other, TDuration::Zero());
}

void TAgentContainer::Mul(const TRecord& record) {
    MulTTL(record, TDuration::Zero());
}

void TAgentContainer::MulTTL(const TAgentContainer& other, const TDuration ttl) {
    TInstant ttlInstant, expireTime;
    CalcTimes(ttl, ttlInstant, expireTime);

    TMaybe<TLightWriteGuard> selfWg;
    TMaybe<TLightReadGuard> otherRg;
    if (&other < this) {
        otherRg.ConstructInPlace(other.Mutex);
        selfWg.ConstructInPlace(Mutex);
    } else {
        selfWg.ConstructInPlace(Mutex);
        otherRg.ConstructInPlace(other.Mutex);
    }
    for (const auto& p: other.Storage) {
        auto value = p.second.GetValue();
        MulInternal(p.first, value, expireTime, ttlInstant);
    }
}

void TAgentContainer::MulTTL(const TRecord& record, const TDuration ttl) {
    TInstant ttlInstant, expireTime;
    CalcTimes(ttl, ttlInstant, expireTime);

    const auto& values = record.GetValues();
    TLightWriteGuard wg(Mutex);
    for (const auto& p: values) {
        auto value = p.second.GetValue();
        MulInternal(p.first, value, expireTime, ttlInstant);
    }
}

size_t TAgentContainer::Trim(const size_t targetSize) {
    TLightWriteGuard wg(Mutex);
    // NOTE(rocco66): trim used in push, there is new signals only
    auto& newAcc = Storage.GetNewAccumulators();
    if (newAcc.size() <= targetSize) {
        return 0;
    }

    const size_t toRemove = newAcc.size() - targetSize;
    auto it = newAcc.begin();
    for (size_t i = 0; i < toRemove; ++i) {
        auto prev = it;
        ++it;
        newAcc.erase(prev);
    }
    return toRemove;
}

void TAgentContainer::MulInternal(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value,
    const TInstant expireTime, const TInstant ttl)
{
    TAgentWrappedAccumulator* acc = Storage.GetAccumulator(name);
    if (acc == nullptr) {
        return;
    }
    acc->Mul(value, expireTime, ttl);
}

size_t TAgentContainer::Len() const noexcept {
    TLightReadGuard rg(Mutex);
    return Storage.Len();
}

void TAgentContainer::Clean(const TInstant explicitNow) {
    TLightWriteGuard wg(Mutex);
    CleanMap(Storage.GetNewAccumulators(), explicitNow);
    CleanMap(Storage.GetOldAccumulators(), explicitNow);
}

void TAgentContainer::Clean() {
    Clean(TInstant::Now());
}

void TAgentContainer::Process(NZoom::NRecord::ISignalValueCallback& callback) const {
    TLightReadGuard rg(Mutex);
    callback.SetObjectsCount(Storage.Len());
    for (const auto& p: Storage) {
        auto value = p.second.GetValue();
        callback.OnSignalValue(p.first, value);
    }
}
