#include "group.h"

using namespace NZoom::NContainers;
using namespace NZoom::NAccumulators;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;

namespace {
    static constexpr TDuration DEFAULT_SIGNAL_TTL_VALUE = TDuration::Seconds(30);

    template <typename TMap>
    void CleanMap(TMap& map, const TInstant now) {
        for (auto it = map.begin(); it != map.end();) {
            auto prev = it;
            ++it;
            if (prev->second.IsExpired(now)) {
                map.erase(prev);
            } else {
                prev->second.Clean();
            }
        }
    }
}


TWrappedAccumulator::TWrappedAccumulator(NZoom::NAccumulators::EAccumulatorType type)
    : Accumulator(type)
{
}

void TWrappedAccumulator::Mul(const TValueRef& value, const TInstant expireTime) {
    Accumulator.Mul(value);
    ExpireTime = expireTime;
}


bool TWrappedAccumulator::IsExpired(const TInstant now) const noexcept {
    return now > ExpireTime;
}

void TWrappedAccumulator::Clean() {
    Accumulator.Clean();
}

NZoom::NValue::TValueRef TWrappedAccumulator::GetValue() const {
    return Accumulator.GetValue();
}

TGroupContainer::TGroupContainer(const NZoom::NYasmConf::TTypeConf& conf)
    : Storage(conf.GetSignals(),
        conf.GetPatterns(), EAggregationMethod::Group)
{
}

void TGroupContainer::Mul(const TGroupContainer& other, NZoom::NValue::TMetricManager& metricManager) {
    const TInstant expireTime = TInstant::Now() + DEFAULT_SIGNAL_TTL_VALUE;
    for (const auto& p: other.Storage) {
        auto value = p.second.GetValue();
        MulInternal(p.first, value, expireTime, metricManager);
    }
}

void TGroupContainer::Mul(const TRecord& record, NZoom::NValue::TMetricManager& metricManager) {
    const TInstant expireTime = TInstant::Now() + DEFAULT_SIGNAL_TTL_VALUE;
    const auto& values = record.GetValues();
    for (const auto& p: values) {
        auto value = p.second.GetValue();
        MulInternal(p.first, value, expireTime, metricManager);
    }
}

void TGroupContainer::MulInternal(const NZoom::NSignal::TSignalName& name, const NZoom::NValue::TValueRef& value,
    const TInstant expireTime, NZoom::NValue::TMetricManager& metricManager)
{
    TWrappedAccumulator* acc = Storage.GetAccumulator(name);
    if (acc == nullptr) {
        return;
    }
    acc->Mul(value, expireTime);
    value.Update(metricManager);
}

size_t TGroupContainer::Len() const noexcept {
    return Storage.Len();
}

void TGroupContainer::Clean(const TInstant explicitNow) {
    CleanMap(Storage.GetNewAccumulators(), explicitNow);
    CleanMap(Storage.GetOldAccumulators(), explicitNow);
}

void TGroupContainer::Clean() {
    Clean(TInstant::Now());
}

void TGroupContainer::Process(NZoom::NRecord::ISignalValueCallback& callback) const {
    callback.SetObjectsCount(Storage.Len());
    for (const auto& p: Storage) {
        auto value = p.second.GetValue();
        callback.OnSignalValue(p.first, value);
    }
}
