#include "group.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace NZoom::NContainers;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;
using namespace NZoom::NYasmConf;


Y_UNIT_TEST_SUITE(TZoomTGroupContainerTest) {

    class TStorageCallback: public ISignalValueCallback {
    private:
        THashMap<TSignalName, TValue> Values;

    public:
        void SetObjectsCount(const size_t count) final {
            Values.reserve(count);
        }

        void OnSignalValue(const TSignalName& name, const TValueRef& value) final {
            Values.emplace(name, value);
        }

        const THashMap<TSignalName, TValue>& GetValues() const noexcept {
            return Values;
        }
    };


    Y_UNIT_TEST(NewSignalGrouping) {
        TYasmConf yasmConf = TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {\"netstat-ibytes\": [false,[\"summ\",true]]}, \"patterns\": {}, \"periods\": {\"netstat-ibytes\": \"summ\"}}}}"));

        TVector <std::pair<TSignalName, TValue>> dataVec;
        dataVec.emplace_back(TString("portoinst-max_rss_gb_tvvv"), TValue(0.04380035400390625, 1));
        TRecord record1(std::move(dataVec));
        TMetricManager metricManager;

        TGroupContainer container(yasmConf.GetTypeConf("common", true));
        container.Mul(record1, metricManager);

        UNIT_ASSERT_VALUES_EQUAL(container.Len(), 1);
        TStorageCallback callback;
        container.Process(callback);

        UNIT_ASSERT_VALUES_EQUAL(callback.GetValues().size(), 1);
    }

    Y_UNIT_TEST(SignalsCleanup) {
        TYasmConf yasmConf = TYasmConf::FromString(TStringBuf("{\"conflist\": {\"common\": {\"signals\": {\"accesslog-2xx\": [false,[\"summ\",true]]}, \"patterns\": {}, \"periods\": {\"accesslog-2xx\": \"summ\"}}}}"));

        TVector <std::pair<TSignalName, TValue>> dataVec;
        dataVec.emplace_back(TString("portoinst-max_rss_gb_tvvv"), TValue(0.04380035400390625, 1));
        dataVec.emplace_back(TString("accesslog-2xx"), TValue(1.0));
        TRecord record1(std::move(dataVec));
        TMetricManager metricManager;

        const TInstant now = TInstant::Now();
        TGroupContainer container(yasmConf.GetTypeConf("common", true));
        container.Mul(record1, metricManager);
        UNIT_ASSERT_VALUES_EQUAL(container.Len(), 2);

        {
            TStorageCallback callback;
            container.Process(callback);
            const auto& values = callback.GetValues();
            UNIT_ASSERT_VALUES_EQUAL(values.size(), 2);
            UNIT_ASSERT_EQUAL(values.at(TSignalName(TStringBuf("accesslog-2xx"))), TValue(1.0));
            UNIT_ASSERT_EQUAL(values.at(TSignalName(TStringBuf("portoinst-max_rss_gb_tvvv"))), TValue(0.04380035400390625, 1));
        }
        container.Clean(now);

        {
            TStorageCallback callback;
            container.Process(callback);
            const auto& values = callback.GetValues();
            UNIT_ASSERT_VALUES_EQUAL(values.size(), 2);
            UNIT_ASSERT_EQUAL(values.at(TSignalName(TStringBuf("accesslog-2xx"))), TValue(0.0));
            UNIT_ASSERT_EQUAL(values.at(TSignalName(TStringBuf("portoinst-max_rss_gb_tvvv"))), TValue(0.0, 0));
        }
        container.Mul(record1, metricManager);
        {
            TStorageCallback callback;
            container.Process(callback);
            const auto& values = callback.GetValues();
            UNIT_ASSERT_VALUES_EQUAL(values.size(), 2);
            UNIT_ASSERT_EQUAL(values.at(TSignalName(TStringBuf("accesslog-2xx"))), TValue(1.0));
            UNIT_ASSERT_EQUAL(values.at(TSignalName(TStringBuf("portoinst-max_rss_gb_tvvv"))), TValue(0.04380035400390625, 1));
        }
        container.Clean(TInstant::Now() + TDuration::Seconds(120));
        {
            TStorageCallback callback;
            container.Process(callback);
            const auto& values = callback.GetValues();
            UNIT_ASSERT_VALUES_EQUAL(values.size(), 0);
        }
    }

}
