#include "integrate.h"

#include <util/string/cast.h>

using namespace NZoom::NContainers;
using namespace NZoom::NAccumulators;
using namespace NZoom::NRecord;
using namespace NZoom::NSignal;
using namespace NZoom::NValue;


TIncompatibleSizeException::TIncompatibleSizeException(const TSourceLocation& sl, const size_t first, const size_t second)
    : std::invalid_argument(
        TString::Join(sl.File, ':', ToString(sl.Line), ": Incompatible size: ",
            ToString(first), " vs ", ToString(second)).c_str())
{
}


TTagMerger::TTagMerger(const NZoom::NYasmConf::TTypeConf& conf, const TVector<TSignalName>& signals, const size_t size) {
    for (auto signal: signals) {
        EmplaceSignal(conf, std::move(signal), size);
    }
}

TTagMerger::TTagMerger(const NZoom::NYasmConf::TTypeConf& conf, INameIterator& signals, const size_t size) {
    for (;signals.IsValid(); signals.Next()) {
        TSignalName signal(signals.GetCurrent());
        EmplaceSignal(conf, std::move(signal), size);
    }

    LeftBorder = 0;
    RightBorder = size;
}

void TTagMerger::EmplaceSignal(const NZoom::NYasmConf::TTypeConf& conf, TSignalName&& signal, const size_t size) {
    const TAggregationRules* aggregationRulesPtr;

    if (signal.IsOld()) {
        aggregationRulesPtr = conf.GetRules(signal);
        if (aggregationRulesPtr == nullptr) {
            aggregationRulesPtr = conf.GetPatterns()->GetMatchedRule(signal.GetName());
        }
        if (aggregationRulesPtr == nullptr) {
            return;
        }
    } else {
        aggregationRulesPtr = signal.GetAggregationRules();
    }
    Accumulators.emplace(signal,
        TAccumulatorsArray(aggregationRulesPtr->GetAccumulatorType(EAggregationMethod::MetaGroup), size));
}

void TTagMerger::MulSignalValues(const NZoom::NSignal::TSignalName& signal, const TVector<NZoom::NValue::TValue>& values,
                                 const ssize_t offset) {
    const auto it = Accumulators.find(signal);
    if (it == Accumulators.end()) {
        return;
    }
    TAccumulatorsArray& accumulators = it->second;

    // remember borders where all records exists
    if (offset >= 0) {
        if (!accumulators.Empty() && offset >= ssize_t(accumulators.Len())) {
            throw TIncompatibleSizeException(__LOCATION__, offset, accumulators.Len());
        }
        LeftBorder = Max<size_t>(LeftBorder, offset);
        RightBorder = Min<size_t>(RightBorder, values.size() + offset);

    } else {
        if (-offset >= ssize_t(values.size())) {
            throw TIncompatibleSizeException(__LOCATION__, -offset, values.size());
        }
        RightBorder = Min<size_t>(RightBorder, values.ysize() + offset);
    }

    const size_t leftOffset = (offset >= 0) ? offset : 0;
    const size_t rightOffset = (offset < 0) ? -offset : 0;

    auto innerIt = accumulators.begin() + leftOffset;
    auto valuesIt = values.cbegin() + rightOffset;

    while (innerIt != accumulators.end() && valuesIt != values.cend()) {
        innerIt->Mul(*valuesIt);
        ++innerIt;
        ++valuesIt;
    }
}

void TTagMerger::MulContinuousRecord(const TContinuousRecord& record, const ssize_t offset) {
    for (const TNamedSeries& v: record.GetValue()) {
        MulSignalValues(v.GetName(), v.GetValues(), offset);
    }
}

void TTagMerger::Process(IContinuousSignalCallback& callback) const {
    callback.SetSignalsCount(Accumulators.size());
    for (const auto& p: Accumulators) {
        IContinuousValueCallback* valuesCallbackPtr = callback.OnSignal(p.first, p.second.Len());
        if (valuesCallbackPtr == nullptr) {
            continue;
        }
        valuesCallbackPtr->Start();
        for (const auto& acc: p.second) {
            valuesCallbackPtr->OnValue(acc.GetValue());
        }
        valuesCallbackPtr->Finish();
    }
}

size_t TTagMerger::GetLeftBorder() const noexcept {
    return LeftBorder;
}

size_t TTagMerger::GetRightBorder() const noexcept {
    return RightBorder;
}
